/*****************************************************************************
 *
 *                                  PNGReader.java 
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jvg;

import java.io.*;
import java.awt.*;
import java.util.Vector;
import java.awt.image.ImageObserver;
import java.awt.image.MemoryImageSource;
import java.awt.image.IndexColorModel;
import java.util.zip.*;

import fi.faidon.util.ByteArrayConversion;

/**
 * Class for reading PNG image files and creating corresponding Java AWT 
 * images.
 *
 * @author Kary FR&Auml;MLING 7/6/1998
 */
public class PNGReader extends VectorImageProducer
{

	//--------------------------------------------------------------------------------------
	// Public constants.
	//--------------------------------------------------------------------------------------

	//--------------------------------------------------------------------------------------
	// Private constants.
	//--------------------------------------------------------------------------------------

	private final boolean	OBLIGATORY_REGISTRATION = false;

	private final byte[]	PNG_IDENTIFICATION_BYTES = {(byte)137, (byte)80, (byte)78, (byte)71, (byte)13, (byte)10, (byte)26, (byte)10};
	private final int		CHUNK_LENGTH_SIZE = 4;
	private final int		CHUNK_TYPE_SIZE = 4;
	private final int		CHUNK_CRC_SIZE = 4;
	private final int		IHDR_DATA_SIZE = 13;
	private final int		INFLATE_BUF_SIZE = 512;
	private final byte		COLOR_TYPE_PALETTE_USED = 1;
	private final byte		COLOR_TYPE_COLOR_USED = 2;
	private final byte		COLOR_TYPE_USE_ALPHA_CHANNEL = 4;

	/**
	 * Filter method definitions.
	 */
	private final byte		NO_FILTERING = 0;
	private final byte		SUB_FILTER = 1;
	private final byte		UP_FILTER = 2;
	private final byte		AVERAGE_FILTER = 3;
	private final byte		PAETH_FILTER = 4;

	/**
	 * Chunk codes.
	 */
	private final byte[]	IHDR_CHUNK_TYPE = {(byte)0x49, (byte)0x48, (byte)0x44, (byte)0x52};
	private final byte[]	PLTE_CHUNK_TYPE = {(byte)0x50, (byte)0x4C, (byte)0x54, (byte)0x45}; 
	private final byte[]	IDAT_CHUNK_TYPE = {(byte)0x49, (byte)0x44, (byte)0x41, (byte)0x54};
	private final byte[]	IEND_CHUNK_TYPE = {(byte)0x49, (byte)0x45, (byte)0x4E, (byte)0x44};
	// 4.2.1. bKGD Background Color 
	// 4.2.2. cHRM Primary Chromaticities and White Point 
	// 4.2.3. gAMA Image Gamma 
	// 4.2.4. hIST Image Histogram 
	// 4.2.5. pHYs Physical Pixel Dimensions 
	// 4.2.6. sBIT Significant Bits 
	// 4.2.7. tEXt Textual Data 
	// 4.2.8. tIME Image Last-Modification Time 
	private final byte[]	tRNS_CHUNK_TYPE = {(byte)0x74, (byte)0x52, (byte)0x4E, (byte)0x53};	// Transparency 
	// 4.2.10. zTXt Compressed Textual Data 

	//--------------------------------------------------------------------------------------
	// Public fields.
	//--------------------------------------------------------------------------------------

	//--------------------------------------------------------------------------------------
	// Private fields.
	//--------------------------------------------------------------------------------------

	private File		pictureFile;
	private FileInputStream		pictureFIS;
	private BufferedInputStream	pictureBIS;
	private Image		pngImage;
	private boolean		verbose;
	private int			width;
	private int			height;
	private int			bitDepth;
	private int			colorType;
	private int			compressionMethod;
	private int			filterMethod;
	private int			interlaceMethod;
//	private Rectangle	pictFrame;
	private int			version;
	private Graphics	drawGraphics;
	private Component	drawComponent;
	private boolean		hdrIsRead;
	private int			pixmapSize;
	private byte[]		rawPixmap;
	private int			pixmapX, pixmapY;
	private int			bytesPerPixel;
	private int			destImageBPP;
	private boolean		useAlpha;
	private boolean		isGrayScale;
	private boolean		needsPalette;
	private boolean		useIndexColorModel;
	private IndexColorModel		indexCModel;
	private int			maxColorValue;

	//--------------------------------------------------------------------------------------
	// Public methods.
	//--------------------------------------------------------------------------------------

	//=============================================================================
	// Constructor
	//=============================================================================
	/**
	 * Initialize stuff.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public PNGReader()
	{
		// Registration check.
		/*
		if ( OBLIGATORY_REGISTRATION ) {
			fi.faidon.protection.SerialNumberManager sm = new fi.faidon.protection.SerialNumberManager();
			if ( !sm.verifyCurrentPackage() ) {
				System.err.println(fi.faidon.protection.SerialNumberManager.STR_REG_VERIFICATION_FAILED);
				System.err.println("Exiting...");
				System.exit(1);
			}
		}
		*/
		hdrIsRead = false;
		verbose = false;
		useAlpha = false;
		isGrayScale = false; 
		useIndexColorModel = false;
	}

	//=============================================================================
	// getWidth {implement VectorImageProducer}
	//=============================================================================
	/**
	 * Implements VectorImageProducer getWidth(). If we don't know the frame size 
	 * yet, then read it in.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public int getWidth()
	{
		if ( !hdrIsRead ) {
			getFrame();
		}
		return width;
	}

	//=============================================================================
	// getHeight {implement VectorImageProducer}
	//=============================================================================
	/**
	 * Implements VectorImageProducer getHeight(). If we don't know the frame size 
	 * yet, then read it in. 
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public int getHeight() 
	{
		if ( !hdrIsRead ) {
			getFrame();
		}
		return height;
	}

	//=============================================================================
	// getImage
	//=============================================================================
	/**
	 * Return the image. 
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public Image getImage() 
	{
		if ( pngImage == null ) {
			readPNGFile();
		}
		return pngImage;
	}

	//=============================================================================
	// playIt {implement VectorImageProducer}
	//=============================================================================
	/**
	 * Implements VectorImageProducer drawIt(). Read in the whole Pixmap, 
	 * create the image and draw it.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public void playIt(Graphics g, Component c) 
	{
		drawGraphics = g;
		drawComponent = c;
		if ( readPNGFile() && pngImage != null ) {
			drawGraphics.drawImage(pngImage, 0, 0, drawComponent);
		}
	}

	//=============================================================================
	// checkImage {implement VectorImageProducer}
	//=============================================================================
	/**
	 * Implements VectorImageProducer drawIntoImage(). Return the status of the 
	 * drawing launched by drawIntoImage. NOT IMPLEMENTED YET!. 
	 *
	 * @author Kary FR&Auml;MLING 4/4/1998
	 */
	//=============================================================================
	public int checkImage() 
	{ 
		return ImageObserver.ALLBITS; 
	}

	//=============================================================================
	// setImageFile
	//=============================================================================
	/**
	 * Sets the path of the image file to use. 
	 *
	 * @author Kary FR&Auml;MLING 31/3/1998
	 */
	//=============================================================================
	public void setImageFile(String filePath)
	{
		// Create a File object on it straight away.
		pictureFile = new File(filePath);

		// Reset some stuff to force a re-read.
		hdrIsRead = false;
	}

	//=============================================================================
	// setVerbose
	//=============================================================================
	/**
	 * Sets verbose to the state indicated. 
	 *
	 * @author Kary FR&Auml;MLING 4/4/1998
	 */
	//=============================================================================
	public void setVerbose(boolean state)
	{
		verbose = state;
	}

	//=============================================================================
	// isVerbose
	//=============================================================================
	/**
	 * Return if verbose or not. 
	 *
	 * @author Kary FR&Auml;MLING 4/4/1998
	 */
	//=============================================================================
	public boolean isVerbose(boolean state) { return verbose; }

	//=============================================================================
	// readPNGFile
	//=============================================================================
	/**
	 * Open and read the PNG file. Create the corresponding MemoryImage.
	 * If "verbose" is true, the chunks read are listed on stdout.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	public boolean readPNGFile()
	{
		int		i;

		// Get input streams on the picture file.
		if ( !getInputStreams() ) return false;

		// Read in header information.
		if ( !readHeader(pictureBIS) ) {
			if ( verbose) System.out.println("Error parsing file!");
			closeInputStreams();
			return false;
		}

		// Get all the rest.
		if ( !readChunks(pictureBIS) ) {
			closeInputStreams();
			if ( verbose) System.out.println("Error parsing file!");
			return false; 
		}

		// Close file streams.
		closeInputStreams();

		// File read OK, return true.
		if ( verbose ) System.out.println("Finished reading file!");

		return true;
	}

	//--------------------------------------------------------------------------------------
	// Private methods.
	//--------------------------------------------------------------------------------------

	//=============================================================================
	// getFrame
	//=============================================================================
	/**
	 * Open and read the frame size of the image.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	private boolean getFrame()
	{
		// Get input streams on the picture file.
		if ( !getInputStreams() ) return false;

		// Read in header information.
		if ( !readHeader(pictureBIS) ) {
			if ( verbose) System.out.println("Error parsing file!");
			closeInputStreams();
			return false;
		}

		// Close file streams.
		closeInputStreams();

		// File read OK, return true.
		return true;
	}

	//=============================================================================
	// getInputStreams
	//=============================================================================
	/**
	 * Initialize the input streams on the PICT file object.
	 *
	 * @author Kary FR&Auml;MLING 31/3/1998
	 */
	//=============================================================================
	private boolean getInputStreams()
	{
		// Test that the file exists.
		if ( pictureFile == null || !pictureFile.exists() ) return false;

		// Open the file.
 		try { 
			pictureFIS = new FileInputStream(pictureFile);
			pictureBIS = new BufferedInputStream(pictureFIS);
 		} catch (IOException e) {
 			return false;
 		}

		// Everything OK, return true.
		return true;
	}	

	//=============================================================================
	// closeInputStreams
	//=============================================================================
	/**
	 * Close all input streams used.
	 *
	 * @author Kary FR&Auml;MLING 4/4/1998
	 */
	//=============================================================================
	private void closeInputStreams()
	{
		try {
			if ( pictureBIS != null ) pictureBIS.close(); 
			if ( pictureFIS != null ) pictureFIS.close(); 
		} catch ( IOException e ) {}
	}

	//=============================================================================
	// readHeader
	//=============================================================================
	/**
	 * Read the header chunk. This has to be the first chunk in the file 
	 * after the PNG identification bytes.
	 *
	 * @author Kary FR&Auml;MLING 31/3/1998
	 */
	//=============================================================================
	private boolean readHeader(BufferedInputStream bis)
	{
		int		i;
		int		chunk_data_len;
		byte[]	buf;
		byte[]	hdr_buf = new byte[CHUNK_LENGTH_SIZE + CHUNK_TYPE_SIZE + IHDR_DATA_SIZE + CHUNK_CRC_SIZE];

		// Verify that we have a correct stream.
		if ( bis == null ) return false;

		// Catch all read errors.
		try {

			// Get first bytes and verify that we indeed have a PNG file.
			buf = new byte[PNG_IDENTIFICATION_BYTES.length];
			fillByteBuf(bis, buf, 0, buf.length);
			for ( i = 0 ; i < buf.length && i < PNG_IDENTIFICATION_BYTES.length ; i++ ) {
				if ( buf[i] != PNG_IDENTIFICATION_BYTES[i] ) {
					if ( verbose ) System.out.println("Not valid PNG file identifier!");
					return false;
				}
			}

			// Read first chunk length, type, data and CRC. This MUST be a IHDR chunk.
			fillByteBuf(bis, hdr_buf, 0, hdr_buf.length);
			chunk_data_len = ByteArrayConversion.bytesLEasUINT(hdr_buf, 0, CHUNK_LENGTH_SIZE);
			if ( chunk_data_len != IHDR_DATA_SIZE ) {
				if ( verbose ) System.out.println("Wrong IHDR data size!");
				return false;
			}
			for ( i = 0 ; i < CHUNK_TYPE_SIZE ; i++ ) {
				if ( hdr_buf[i + 4] != IHDR_CHUNK_TYPE[i] ) {
					if ( verbose ) System.out.println("File does not begin with an IHDR chunk!");
					return false;
				}
			}

			width = ByteArrayConversion.bytesLEasUINT(hdr_buf, 8, 12);
			height = ByteArrayConversion.bytesLEasUINT(hdr_buf, 12, 16);
			bitDepth = ByteArrayConversion.bytesLEasUINT(hdr_buf, 16, 17);
			colorType = ByteArrayConversion.bytesLEasUINT(hdr_buf, 17, 18);
			compressionMethod = ByteArrayConversion.bytesLEasUINT(hdr_buf, 18, 19);
			filterMethod = ByteArrayConversion.bytesLEasUINT(hdr_buf, 19, 20);
			interlaceMethod = ByteArrayConversion.bytesLEasUINT(hdr_buf, 20, 21);
			if ( verbose ) System.out.println("IHDR, width: " + width + ", height: " + height + 
					", bitDepth: " + bitDepth + ", colorType: " + colorType + ", compressionMethod: " + compressionMethod + 
					", filterMethod: " + filterMethod + ", interlaceMethod: " + interlaceMethod);

			// Determine bytes/pixel depending on the color type. We support 
			// both 8 and 16 bit depth. 
			destImageBPP = 3;
			if ( (colorType&COLOR_TYPE_COLOR_USED) > 0 ) {
				bytesPerPixel = 3*(bitDepth/8);
			} 
			else {
				// Gray scale image. 
				isGrayScale = true;
				bytesPerPixel = (int) Math.max(1, bitDepth/8);
			}

			// Palette or not? This is only valid for color images, so if 
			// we use a palette, colorType should always be 3 in fact. 
			if ( (colorType&COLOR_TYPE_PALETTE_USED) > 0 ) {
				bytesPerPixel = 1;
				destImageBPP = 1;
				needsPalette = true;
				isGrayScale = false;
			}

			// See if we have a alpha channel. We suppose that the bit depth of the 
			// alpha channel is the same as that of the color samples, but this 
			// should be verified. Only 8 or 16 are supported bit depths when 
			// using an alpha channel. 
			if ( (colorType&COLOR_TYPE_USE_ALPHA_CHANNEL) > 0 ) {
				bytesPerPixel += bitDepth/8;
				useAlpha = true;
			}

			// Calculate the maximal possible color value. 
			maxColorValue = ((int) Math.pow(2, bitDepth)) - 1;

			// Calculate the byte buffer size needed. The destination image bytes/pixel 
			// is always 3 or 4 if there is an alpha channel. There should never be an alpha 
			// channel for palette images. 
			if ( useAlpha ) destImageBPP++;
			pixmapSize = width*height*destImageBPP;

			// We should check the CRC too, but there will certainly be read errors 
			// later if there is a problem. 

 		} catch (IOException e) { 
			closeInputStreams();
			return false;
		}

		// Header is read.
		hdrIsRead = true;

		// Everything OK, return true.
		return true;
	}

	//=============================================================================
	// readChunks
	//=============================================================================
	/**
	 * Read all the chunks and create the corresponding image.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	private boolean readChunks(BufferedInputStream bis)
	{
		int		i, j;
		int		chunk_data_len;
		int		bytes_read;
		int		inflate_cnt;
		int		rgb_value_cnt;
		int		r, g, b, alpha;
		int		scanline_fltr;
		int		pixmap_offset;
		int		bit_shift_cnt;
		byte[]	buf;
		byte[]	chunk_start_buf = new byte[CHUNK_LENGTH_SIZE + CHUNK_TYPE_SIZE];
		byte[]	crc_buf = new byte[CHUNK_CRC_SIZE];
		byte[]	data_buf = new byte[INFLATE_BUF_SIZE];
		byte[]	inflate_buf = new byte[INFLATE_BUF_SIZE];
		boolean	end_chunk_found = false;
		boolean	is_first_line = true;
		Inflater	infl = new Inflater();

		// Verify that we have a correct stream.
		if ( bis == null ) return false;

		// Set up pixel buffer for the RGB values.
		rawPixmap = new byte[pixmapSize];
		pixmapX = pixmapY = 0;

		// RGB converted byte counter is 0.
		rgb_value_cnt = 0;
		pixmap_offset = 0;
		scanline_fltr = NO_FILTERING;

		// Catch all read errors.
		try {
			// Go until we find an end chunk.
			while ( !end_chunk_found ) {
				// Read chunk length and type.
				fillByteBuf(bis, chunk_start_buf, 0, chunk_start_buf.length);
				chunk_data_len = ByteArrayConversion.bytesLEasUINT(chunk_start_buf, 0, CHUNK_LENGTH_SIZE);
				if ( verbose ) System.out.println("Chunk data length: " + chunk_data_len); 

				// React according to the data type. 
				switch ( chunk_start_buf[4] ) {
				case (byte)0x49:		// 'I'.
					// IDAT.
					if ( chunk_start_buf[5] == IDAT_CHUNK_TYPE[1] && chunk_start_buf[6] == IDAT_CHUNK_TYPE[2] && 
							chunk_start_buf[7] == IDAT_CHUNK_TYPE[3] ) {
						if ( verbose ) System.out.println("IDAT chunk."); 

						// Read in pixmap data. 
						for ( bytes_read = 0 ; bytes_read < chunk_data_len ; bytes_read += INFLATE_BUF_SIZE ) {
							fillByteBuf(bis, data_buf, 0, (int) Math.min(data_buf.length, chunk_data_len - bytes_read));
							infl.setInput(data_buf, 0, (int) Math.min(data_buf.length, chunk_data_len - bytes_read));
							while ( !infl.needsInput() ) {
								try {
									// Inflate some.
									inflate_cnt = infl.inflate(inflate_buf);

									// Treat the inflated bytes.
									for ( i = 0 ; i < inflate_cnt ; i++ ) {
										// If we have a complete scanline, then 
										// get the new filter value. Special treatment for first line.
										if ( is_first_line ) {
											scanline_fltr = inflate_buf[i];
											is_first_line = false;
										}
										else if ( rgb_value_cnt == width*bytesPerPixel ) {
											pixmapX = 0;
											pixmapY++;
											scanline_fltr = inflate_buf[i];
											rgb_value_cnt = 0;
										}
										else {
											switch ( rgb_value_cnt%bytesPerPixel  )	{
											// Red, gray or indexed value. The cases treated here are: 
											// - Color image: Treat red value at bit depth 8 or 16.	16 not implemented yet. 
											// - Gray scale image: Treat one value of any bit depth. Only 8 implemented yet. 
											// - Palette index: One or more index values at all supported bit 
											//   depths (1,2,4,8). 
											case 0:
												// If we are not using a palette, then we get the red value. 
												// If we are using a palette, then we redistribute the bits 
												// on the right number of pixels.
												r = getFltrValue(scanline_fltr, inflate_buf[i]&0xFF, rawPixmap, pixmap_offset);
												if ( !useIndexColorModel ) {
													rawPixmap[pixmap_offset++] = (byte) (r&0xFF);

													// If it is a grayscale image, then we set g and b to same value. 
													if ( isGrayScale ) {
														rawPixmap[pixmap_offset++] = (byte) (r&0xFF);
														rawPixmap[pixmap_offset++] = (byte) (r&0xFF);
													}
												}
												else {
													// Use all the bits of the byte.
													for ( bit_shift_cnt = 0 ; bit_shift_cnt < 8/bitDepth ; bit_shift_cnt++ ) {
														switch ( bitDepth ) {
														case 1:
															rawPixmap[pixmap_offset++] = (byte) (r&0x1);
															r >>= 1;
															break;
														case 2:
															rawPixmap[pixmap_offset++] = (byte) (r&0x3);
															r >>= 2;
															break;
														case 4:
															rawPixmap[pixmap_offset++] = (byte) (r&0xF);
															r >>= 4;
															break;
														default:
															rawPixmap[pixmap_offset++] = (byte) (r&0xFF);
														}
													} 
												}
												break;

											// Green or alpha value. The cases treated here are: 
											// - Color image: green value at bit depth 8. 
											// - Grayscale image: alpha value for bit depth 8. 
											case 1:	
												g = getFltrValue(scanline_fltr, inflate_buf[i]&0xFF, rawPixmap, pixmap_offset);
												rawPixmap[pixmap_offset++] = (byte) (g&0xFF);
												break;

											// The cases treated here are: 
											// - Color image: blue value at bit depth 8, green at bit depth 16 (not supported yet).
											// - Grayscale image: alpha value for bit depth 16 (not supported yet). 
											case 2: 
												b = getFltrValue(scanline_fltr, inflate_buf[i]&0xFF, rawPixmap, pixmap_offset);
												rawPixmap[pixmap_offset++] = (byte) (b&0xFF);
												break;

											// The cases treated here are: 
											// - Color image: alpha value at bit depth 8.
											case 3: 
												alpha = getFltrValue(scanline_fltr, inflate_buf[i]&0xFF, rawPixmap, pixmap_offset);
												rawPixmap[pixmap_offset++] = (byte) (alpha&0xFF);
												break;

											// The cases treated here are: 
											// - Color image: blue value at bit depth 16 (not supported yet). 
											case 4: 
												break;

											// The cases treated here are: 
											// - Color image: alpha value at bit depth 16 (not supported yet). 
											case 6: 
												break;
											}

											// Go to next pixel if we have the last component.
											if ( rgb_value_cnt%bytesPerPixel == bytesPerPixel - 1 )
												pixmapX++;
											rgb_value_cnt++;
										}
									}
								} catch ( DataFormatException e ) {
									if ( verbose ) System.out.println("DataFormatException: " + e); 
									return false;
								}
							}
						}
					}

					// IEND.
					else if ( chunk_start_buf[5] == IEND_CHUNK_TYPE[1] && chunk_start_buf[6] == IEND_CHUNK_TYPE[2] && 
							chunk_start_buf[7] == IEND_CHUNK_TYPE[3] ) {
						// Finished reading. 
						if ( verbose ) System.out.println("IEND chunk.");
						end_chunk_found = true;
					}
					break;

				case (byte) 0x50:		// 'P'
					// PLTE.
					if ( chunk_start_buf[5] == PLTE_CHUNK_TYPE[1] && chunk_start_buf[6] == PLTE_CHUNK_TYPE[2] && 
							chunk_start_buf[7] == PLTE_CHUNK_TYPE[3] ) {
						// Found a palette chunk. 
						useIndexColorModel = true;
						if ( verbose ) System.out.println("PLTE chunk.");

						// Set up an array with RGB values.
						int nbr_rgb_values = chunk_data_len/3;
						byte[] rgb_buf = new byte[chunk_data_len];

						// Read in the RGB data.
						fillByteBuf(bis, rgb_buf, 0, rgb_buf.length);

						// Create the corresponding IndexColorModel. 
						indexCModel = new IndexColorModel(bitDepth, nbr_rgb_values, rgb_buf, 0, false);
					}
					break;

				case (byte) 0x74:		// 't'
					// tRNS.
					if ( chunk_start_buf[5] == tRNS_CHUNK_TYPE[1] && chunk_start_buf[6] == tRNS_CHUNK_TYPE[2] && 
							chunk_start_buf[7] == tRNS_CHUNK_TYPE[3] ) {
						// Found a transparency value chunk. 
						if ( verbose ) System.out.println("tRNS chunk.");

						// Verify that we have a palette. Since the tRNS chunk should always come 
						// after the PLTE chunk, it is an error if we have none. 
						if ( useIndexColorModel && indexCModel != null ) {
							// Get the alpha values and copy them into a byte array of the 
							// same size as the RGB arrays. We initialize all alpha 
							// values to 255 (opaque) by default. 
							int map_size = indexCModel.getMapSize();
							byte[] alpha_buf = new byte[map_size];
							for ( i = 0 ; i < alpha_buf.length ; i++ ) alpha_buf[i] = (byte) 0xFF;

							// Read in the RGB data as far as needed.
							fillByteBuf(bis, alpha_buf, 0, (int) Math.min(chunk_data_len, alpha_buf.length));

							// Transform the current IndexColorModel to a new one including
							// alpha values. 
							byte[] rbuf = new byte[map_size];
							byte[] gbuf = new byte[map_size]; 
							byte[] bbuf = new byte[map_size];
							indexCModel.getReds(rbuf);
							indexCModel.getGreens(gbuf);
							indexCModel.getBlues(bbuf);
							indexCModel = new IndexColorModel(8, map_size, rbuf, gbuf, bbuf, alpha_buf);
						}
					}
					break;

				default:				// Unknown chunk, read the data + CRC and go on.
					fillByteBuf(bis, new byte[chunk_data_len], 0, chunk_data_len);
					if ( verbose ) System.out.println("Unknown chunk: " + new String(chunk_start_buf, 4, CHUNK_TYPE_SIZE)); 
				}

				// Chunk CRC.
				fillByteBuf(bis, crc_buf, 0, crc_buf.length);

				// We should check the CRC too, but well... . 

			}
 		} catch (IOException e) { 
			if ( verbose ) System.out.println("IOException: " + e); 
			closeInputStreams();
			return false;
		}

		// Pack the RGB array into an int array if we are using a direct color model.
		if ( !useIndexColorModel ) {
			int[] int_pix_buf = new int[width*height];
			for ( i = 0, j = 0 ; i < width*height ; i++, j += destImageBPP ) {
				if ( useAlpha ) {
					int_pix_buf[i] = ((rawPixmap[j]&0xFF)<<16) | ((rawPixmap[j + 1]&0xFF)<<8)
							 | (rawPixmap[j + 2]&0xFF) | ((rawPixmap[j + 3]&0xFF)<<24);
				}
				else {
					int_pix_buf[i] = 0xFF000000 | ((rawPixmap[j]&0xFF)<<16) | ((rawPixmap[j + 1]&0xFF)<<8)
							 | (rawPixmap[j + 2]&0xFF);
				}
			}

			// Create the memory image.
			pngImage = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, int_pix_buf, 0, width));
		}
		else {
			// Used the IndexColorModel for creating the image.
			pngImage = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, indexCModel, rawPixmap, 0, width));
		}

		// Everything OK, return true.
		return true;
	}

	//=============================================================================
	// getFltrValue
	//=============================================================================
	/**
	 * Get the pixel value as a function of the current filter, the byte value 
	 * and the surrounding pixels.
	 *
	 * @author Kary FR&Auml;MLING 7/6/1998
	 */
	//=============================================================================
	private int getFltrValue(int filter, int value, byte[] pixBuf, int pixBufOff)
	{
		int		left, upper_left, above;
		int		offset;
		int		average;

		// Get neighbor pixel values.
		offset = pixBufOff - destImageBPP;
		if ( pixmapX <= 0 ) left = 0;
		else left = pixBuf[offset]&0xFF;
		offset = pixBufOff - destImageBPP*width;
		if ( pixmapY <= 0 ) above = 0;
		else above = pixBuf[offset]&0xFF;
		offset -= destImageBPP;
		if ( pixmapX <= 0 || pixmapY <= 0 ) upper_left = 0;
		else upper_left = pixBuf[offset]&0xFF;

		// Calculate value depending on the filter.
		switch ( filter ) {
		case NO_FILTERING:
			return value;
		case SUB_FILTER:
			return ((value&0xFF) + left)&0xFF;
		case UP_FILTER:
			return ((value&0xFF) + above)&0xFF;
		case AVERAGE_FILTER:
			return ((value&0xFF) + ((left + above)/2)&0xFF)&0xFF;
		case PAETH_FILTER:
			return ((value&0xFF) + paethPredictor(left, above, upper_left)&0xFF)&0xFF;
		default: 
			return value;
		}
	}

	//=============================================================================
	// paethPredictor
	//=============================================================================
	/**
	 * Return the Paeth predictor value as a function of the neighbor pixel 
	 * values a, b and c.
	 *
	 * @author Kary FR&Auml;MLING 9/6/1998
	 */
	//=============================================================================
	private int paethPredictor(int a, int b, int c)
	{
		int		p;
		int		pa, pb, pc;

        // a = left, b = above, c = upper left
		p = a + b - c;				// initial estimate
		pa = Math.abs(p - a);		// distances to a, b, c
        pb = Math.abs(p - b);
        pc = Math.abs(p - c);

        // return nearest of a,b,c,
        // breaking ties in order a,b,c.
        if ( pa <= pb && pa <= pc ) {
			return a;
        }
        if ( pb <= pc ) {
			return b;
        }
		return c;
	}

	//=============================================================================
	// fillByteBuf
	//=============================================================================
	/**
	 * Read bytes from the input stream until the requested number of 
	 * bytes has been retreved. This may not always be the case with 
	 * buffered readers, for instance, if their buffer runs empty. 
	 * Throws an IOException if there is a problem somewhere, like running 
	 * into an EOF before finished reading.
	 *
	 * @author Kary FR&Auml;MLING 14/4/1998
	 */
	//=============================================================================
	private void fillByteBuf(InputStream is, byte[] buf, int off, int len)
			throws IOException
	{
		int		bytes_read, byte_cnt = 0;

		while ( len > 0 ) {
			if ( (bytes_read = is.read(buf, off, len)) == -1 ) throw new IOException("Unexpected EOF");
			off += bytes_read;
			len -= bytes_read;
		}
	}

}

