/*****************************************************************************
 *
 *                            GraphicsRecording.java
 *
 * Created by Kary FRAMLING 12/4/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jvg;

import java.io.*;
import java.awt.*;
import java.awt.image.ImageObserver;
import java.util.Vector;
import java.util.zip.*;
import fi.faidon.jis.*;

/**
 * Class for storing graphical operations and the objects related to
 * them.
 *
 * @author Kary FR&Auml;MLING 12/4/1998
 */
public class GraphicsRecording extends VectorImageProducer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final byte[]	IMG_FILE_SIGNATURE = {'J', 'V', 'G', 'F'};
    public static final int		NO_COMPRESSION = 0;
    public static final int		GZIP_COMPRESSION = 1;
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final int	DEF_REC_VECT_SIZE = 128;
    private final int	DEF_IMAGE_VECT_SIZE = 24;
    
    //--------------------------------------------------------------------------------------
    // Private variables.
    //--------------------------------------------------------------------------------------
    private Vector		recordingVector;
    private ImageSerializerVector	imageVector;
    private JVGHeader	header;
    private boolean		verbose;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Construct a new graphics operation with the give operation code.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public GraphicsRecording() {
	// Set up the recording graphics.
	recordingVector = new Vector(DEF_REC_VECT_SIZE);
	
	// Add the current JVG version header.
	recordingVector.addElement(new Version1JVGHeader());
	
	// Add the object that indicates that this package has been used for creating
	// the recording.
	recordingVector.addElement(new FaidonJVGIdentifier());
	
	// Set up image referencing vector.
	imageVector = new ImageSerializerVector(DEF_IMAGE_VECT_SIZE);
	
	// Verbose mode off by default.
	verbose = false;
    }
    
    //=============================================================================
    // isVerbose
    //=============================================================================
    /**
     * Verbose mode or not?.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public boolean isVerbose() {
	return verbose;
    }
    
    //=============================================================================
    // setVerbose
    //=============================================================================
    /**
     * Set or clear verbose mode.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public void setVerbose(boolean isVerbose) {
	verbose = isVerbose;
    }
    
    //=============================================================================
    // addOperation
    //=============================================================================
    /**
     * Add an operation to the recording.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public void addOperation(Object[] opObjects) {
	int		image_index;
	
	// Add all elements of the object array to the recording vector.
	for ( int i = 0 ; i < opObjects.length ; i++ ) {
	    // See if it is an image. If it is, we get its' index in the
	    // vector of images used already (it is added if not there yet)
	    // and we add an Integer object with the index instead.
	    if ( opObjects[i] instanceof Image ) {
		image_index = getStoredImage((Image) opObjects[i]);
		recordingVector.addElement(new Integer(image_index));
	    }
	    else {
		recordingVector.addElement(opObjects[i]);
	    }
	}
    }
    
    //=============================================================================
    // addOperation
    //=============================================================================
    /**
     * Add an operation to the recording. Operation objects are stored in
     * a vector.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public void addOperation(Vector opObjects) {
	int		image_index;
	
	for ( int i = 0 ; i < opObjects.size() ; i++ ) {
	    // See if it is an image. If it is, we get its' index in the
	    // vector of images used already (it is added if not there yet)
	    // and we add an Integer object with the index instead.
	    if ( opObjects.elementAt(i) instanceof Image ) {
		image_index = getStoredImage((Image) opObjects.elementAt(i));
		recordingVector.addElement(new Integer(image_index));
	    }
	    else {
		recordingVector.addElement(opObjects.elementAt(i));
	    }
	}
    }
    
    //=============================================================================
    // getImages
    //=============================================================================
    /**
     * Get an array of all Image objects in the recording. This is useful for
     * doing "checkImage"s on them in order to avoid screen flicker.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public Image[] getImages() {
	Image[]		images = new Image[imageVector.size()];
	
	for ( int i = 0 ; i < imageVector.size() ; i++ )
	    images[i] = ((ImageSerializer) imageVector.elementAt(i)).getImage();
	return images;
    }
    
    //=============================================================================
    // getWidth
    //=============================================================================
    /**
     * Always return -1 for the moment. This information is not too interesting
     * for vector images.
     *
     * @author Kary FR&Auml;MLING 8/5/1998
     */
    //=============================================================================
    public int getWidth() {
	return -1;
    }
    
    
    //=============================================================================
    // getHeight
    //=============================================================================
    /**
     * Always return -1 for the moment. This information is not too interesting
     * for vector images.
     *
     * @author Kary FR&Auml;MLING 8/5/1998
     */
    //=============================================================================
    public int getHeight() {
	return -1;
    }
    
    //=============================================================================
    // playIt
    //=============================================================================
    /**
     * Play the recorded operations into the given Graphics object. "drawComponent"
     * is the component that the graphics belongs to, necessary for setting the
     * background color and for image drawing (where we need an ImageObserver).
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public void playIt(Graphics g, Component drawComponent) {
	int		i, object_ind;
	int		img_index;
	Point	pen_position = new Point(0, 0);
	Point	tmp_point;
	Rectangle	tmp_rect;
	Rectangle	cur_rect = new Rectangle();
	Rectangle	src_rect, dst_rect;
	String	cur_text = "";
	Font	cur_font = Font.decode("Times");
	Polygon	cur_poly = new Polygon();
	Integer	cur_img_index = new Integer(0);
	Boolean	a_boolean_obj = new Boolean(false);
	Color	fg_color = Color.black;
	Color	xor_color = Color.white;
	Color	tmp_color;
	Vector	op_objects = new Vector();
	Vector	op_args;
	Image	img;
	Dimension	a_dimension = new Dimension();
	GraphicsOperation	current_op = new GraphicsOperation(GraphicsOperation.NOP);
	
	// Preliminary verifications.
	if ( recordingVector == null ) return;
	
	// Jump over header and stuff and go for the first operation.
	object_ind = 0;
	while ( object_ind < recordingVector.size() &&
	!(recordingVector.elementAt(object_ind) instanceof GraphicsOperation) ) {
	    
	    // Is it a header?
	    if ( recordingVector.elementAt(object_ind) instanceof JVGHeader )
		header = (JVGHeader) recordingVector.elementAt(object_ind);
	    
	    // Is it a Faidon created JVG?
	    //			if ( recordingVector.elementAt(object_ind) instanceof FaidonJVGIdentifier )
	    //				System.out.println("We got the FaidonJVGIdentifier");
	    
	    object_ind++;
	}
	
	// Go ahead and play it.
	try {
	    while ( object_ind < recordingVector.size() ) {
		
		// Get next operation vector.
		object_ind = getOpVector(recordingVector, object_ind, op_objects);
		
		// Supplementary tests: if op_objects is empty, there's no more operations.
		if ( op_objects.size() == 0 ) return;
		
		// Allright, no problem, get the operation.
		current_op = (GraphicsOperation) op_objects.elementAt(0);
		
		// See what it was and react accordingly.
		switch ( current_op.getOperation() ) {
		    
		    case GraphicsOperation.CLEAR_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.clearRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("clearRect: " + cur_rect);
			break;
			
		    case GraphicsOperation.CLIP_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.clipRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("clipRect: " + cur_rect);
			break;
			
		    case GraphicsOperation.COPY_AREA:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get dx, dy.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.copyArea(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, tmp_point.x, tmp_point.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("copyArea: " + cur_rect + ", dx, dy: " + tmp_point);
			break;
			
		    case GraphicsOperation.DRAW_3D_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get Boolean arguments of the operation. If none, then the old value
			// of "raised" is used (false by default).
			op_args = getObjectsOfType(op_objects, 1, a_boolean_obj.getClass());
			if ( op_args.size() > 0 )
			    a_boolean_obj = (Boolean) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.draw3DRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, a_boolean_obj.booleanValue());
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("draw3DRect: " + cur_rect + ", raised: " + a_boolean_obj.booleanValue());
			break;
			
		    case GraphicsOperation.DRAW_ARC:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get angles "startAngle" and "arcAngle", stored as a Point object.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawArc(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, tmp_point.x, tmp_point.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawArc: " + cur_rect + ", angles: " + tmp_point);
			break;
			
		    case GraphicsOperation.DRAW_IMAGE_AT_POINT:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get destimation point, stored as a Point object.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null )
			    g.drawImage(img, tmp_point.x, tmp_point.y, drawComponent);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawImageAtOrigin of image: " + img_index + " at: " + tmp_point);
			break;
			
		    case GraphicsOperation.DRAW_IMAGE_IN_RECT:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get destination rectangle, stored as a Rectangle object.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null )
			    g.drawImage(img, tmp_rect.x, tmp_rect.y, tmp_rect.width, tmp_rect.height, drawComponent);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawImageInRect of image: " + img_index + " in: " + tmp_rect);
			break;
			
		    case GraphicsOperation.DRAW_IMAGE_AT_POINT_WITH_BGCOLOR:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get destination point, stored as a Point object.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			
			// Get background color, stored as a Color object.
			op_args = getObjectsOfType(op_objects, 1, fg_color.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_color = (Color) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null )
			    g.drawImage(img, tmp_point.x, tmp_point.y, tmp_color, drawComponent);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawImageAtPointWithBGColor of image: " + img_index + " at: " + tmp_point + ", bgColor:" + tmp_color);
			break;
			
		    case GraphicsOperation.DRAW_IMAGE_IN_RECT_WITH_BGCOLOR:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get destination rectangle, stored as a Rectangle object.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_rect = (Rectangle) op_args.elementAt(0);
			
			// Get background color, stored as a Color object.
			op_args = getObjectsOfType(op_objects, 1, fg_color.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_color = (Color) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null )
			    g.drawImage(img, tmp_rect.x, tmp_rect.y, tmp_rect.width, tmp_rect.height, tmp_color, drawComponent);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawImageInRectWithBGColor of image: " + img_index + " in: " + tmp_rect + ", bgColor:" + tmp_color);
			break;
			
		    case GraphicsOperation.DRAW_SUBIMAGE_IN_RECT:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get source and destination rectangles, stored as a Rectangle objects.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 2 ) break;
			src_rect = (Rectangle) op_args.elementAt(0);
			dst_rect = (Rectangle) op_args.elementAt(1);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null ) {
			    g.drawImage(
			    img,
			    dst_rect.x,
			    dst_rect.y,
			    dst_rect.x + dst_rect.width,
			    dst_rect.y + dst_rect.height,
			    src_rect.x,
			    src_rect.y,
			    src_rect.x + src_rect.width,
			    src_rect.y + src_rect.height,
			    drawComponent
			    );
			}
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawSubimageInRect of image: " + img_index + " src: " + src_rect + ", dst:" + dst_rect);
			break;
			
		    case GraphicsOperation.DRAW_SUBIMAGE_IN_RECT_WITH_BGCOLOR:
			// Get the image index as an Integer argument of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_img_index.getClass());
			if ( op_args.size() < 1 ) break;
			cur_img_index = (Integer) op_args.elementAt(0);
			img_index = cur_img_index.intValue();
			img = ((ImageSerializer) imageVector.elementAt(img_index)).getImage();
			
			// Get source and destination rectangles, stored as a Rectangle objects.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 2 ) break;
			src_rect = (Rectangle) op_args.elementAt(0);
			dst_rect = (Rectangle) op_args.elementAt(1);
			
			// Get background color, stored as a Color object.
			op_args = getObjectsOfType(op_objects, 1, fg_color.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_color = (Color) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null && drawComponent != null ) {
			    g.drawImage(
			    img,
			    dst_rect.x,
			    dst_rect.y,
			    dst_rect.x + dst_rect.width,
			    dst_rect.y + dst_rect.height,
			    src_rect.x,
			    src_rect.y,
			    src_rect.x + src_rect.width,
			    src_rect.y + src_rect.height,
			    tmp_color,
			    drawComponent
			    );
			}
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawSubimageInRectWithBGColor of image: " + img_index + " src: " + src_rect + ", dst:" + dst_rect + ", bgColor:" + tmp_color);
			break;
			
		    case GraphicsOperation.DRAW_LINE:
			// Get all point arguments of the operation. Break if less than two.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 2 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			pen_position = (Point) op_args.elementAt(1);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawLine(tmp_point.x, tmp_point.y, pen_position.x, pen_position.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("Line from " + tmp_point + " to " + pen_position);
			break;
			
		    case GraphicsOperation.DRAW_OVAL:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawOval(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("Oval: " + cur_rect);
			break;
			
		    case GraphicsOperation.DRAW_POLYGON:
			// Get polygon arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_poly.getClass());
			if ( op_args.size() < 1 ) break;
			cur_poly = (Polygon) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawPolygon(cur_poly);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawPolygon: " + cur_poly);
			break;
			
		    case GraphicsOperation.DRAW_POLYLINE:
			// Get polygon arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_poly.getClass());
			if ( op_args.size() < 1 ) break;
			cur_poly = (Polygon) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawPolyline(cur_poly.xpoints, cur_poly.ypoints, cur_poly.npoints);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawPolyline: " + cur_poly);
			break;
			
		    case GraphicsOperation.DRAW_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawRect: " + cur_rect);
			break;
			
		    case GraphicsOperation.DRAW_ROUND_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get angles "arcWidth" and "arcHeight", stored as a Dimension object.
			op_args = getObjectsOfType(op_objects, 1, a_dimension.getClass());
			if ( op_args.size() < 1 ) break;
			a_dimension = (Dimension) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawRoundRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, a_dimension.width, a_dimension.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawRoundRect: " + cur_rect + ", corner arc: " + a_dimension);
			break;
			
		    case GraphicsOperation.DRAW_STRING:
			// Get all string arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_text.getClass());
			if ( op_args.size() < 1 ) break;
			cur_text = (String) op_args.elementAt(0);
			
			// Get display point of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			pen_position = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.drawString(cur_text, pen_position.x, pen_position.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("drawString: " + cur_text + ", " + pen_position);
			break;
			
		    case GraphicsOperation.FILL_3D_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get Boolean arguments of the operation. If none, then the old value
			// of "raised" is used (false by default).
			op_args = getObjectsOfType(op_objects, 1, a_boolean_obj.getClass());
			if ( op_args.size() > 0 )
			    a_boolean_obj = (Boolean) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fill3DRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, a_boolean_obj.booleanValue());
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fill3DRect: " + cur_rect + ", raised: " + a_boolean_obj.booleanValue());
			break;
			
		    case GraphicsOperation.FILL_ARC:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get angles "startAngle" and "arcAngle", stored as a Point object.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			tmp_point = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fillArc(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, tmp_point.x, tmp_point.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fillArc: " + cur_rect + ", angles: " + tmp_point);
			break;
			
		    case GraphicsOperation.FILL_OVAL:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fillOval(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fillOval: " + cur_rect);
			break;
			
		    case GraphicsOperation.FILL_POLYGON:
			// Get polygon arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_poly.getClass());
			if ( op_args.size() < 1 ) break;
			cur_poly = (Polygon) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fillPolygon(cur_poly);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fillPolygon: " + cur_poly);
			break;
			
		    case GraphicsOperation.FILL_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fillRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fillRect: " + cur_rect);
			break;
			
		    case GraphicsOperation.FILL_ROUND_RECT:
			// Get all rectangle arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Get angles "arcWidth" and "arcHeight", stored as a Dimension object.
			op_args = getObjectsOfType(op_objects, 1, a_dimension.getClass());
			if ( op_args.size() < 1 ) break;
			a_dimension = (Dimension) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.fillRoundRect(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height, a_dimension.width, a_dimension.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("fillRoundRect: " + cur_rect + ", corner arc: " + a_dimension);
			break;
			
		    case GraphicsOperation.SET_CLIP:
			// Get clip rect if any. Break if none (CHANGE THIS TO TREAT SHAPES TOO!).
			op_args = getObjectsOfType(op_objects, 1, cur_rect.getClass());
			if ( op_args.size() < 1 ) break;
			cur_rect = (Rectangle) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.setClip(cur_rect.x, cur_rect.y, cur_rect.width, cur_rect.height);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("SetClip: " + cur_rect);
			break;
			
		    case GraphicsOperation.SET_COLOR:
			// Get the color object. Break if none.
			op_args = getObjectsOfType(op_objects, 1, fg_color.getClass());
			if ( op_args.size() < 1 ) break;
			fg_color = (Color) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.setColor(fg_color);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("setColor: " + fg_color);
			break;
			
		    case GraphicsOperation.SET_FONT:
			// Get the font object. Break if none.
			op_args = getObjectsOfType(op_objects, 1, cur_font.getClass());
			if ( op_args.size() < 1 ) break;
			cur_font = (Font) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.setFont(cur_font);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("setFont: " + cur_font);
			break;
			
		    case GraphicsOperation.SET_PAINT_MODE:
			// Draw if non-null graphics.
			if ( g != null ) g.setPaintMode();
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("setPaintMode");
			break;
			
		    case GraphicsOperation.SET_XOR_MODE:
			// Get the font object. If none, then we use the previous one.
			op_args = getObjectsOfType(op_objects, 1, xor_color.getClass());
			if ( op_args.size() < 1 ) break;
			xor_color = (Color) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.setXORMode(xor_color);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("setXORMode: " + xor_color);
			break;
			
		    case GraphicsOperation.TRANSLATE:
			// Get all point arguments of the operation. Break if none.
			op_args = getObjectsOfType(op_objects, 1, pen_position.getClass());
			if ( op_args.size() < 1 ) break;
			pen_position = (Point) op_args.elementAt(0);
			
			// Draw if non-null graphics.
			if ( g != null ) g.translate(pen_position.x, pen_position.y);
			
			// Text output if verbose mode.
			if ( verbose ) System.out.println("translate: " + pen_position);
			break;
		}
	    }
	} catch ( InvalidObjectException ioe ) { return; }
    }
    
    //=============================================================================
    // checkImage {IMPLEMENT}
    //=============================================================================
    /**
     * Return the status as a function of the state of all images.
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public int checkImage() {
	return ImageObserver.ALLBITS;
    }
    
    //=============================================================================
    // readFromFile
    //=============================================================================
    /**
     * Restore the object variables from the given file.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public synchronized boolean readFromFile(File f) {
	
	FileInputStream		istream;
	ObjectInputStream	p;
	GZIPInputStream		gzstream = null;
	
	try {
	    istream = new FileInputStream(f);
	    
	    // Check file signature.
	    byte[] signature = new byte[IMG_FILE_SIGNATURE.length];
	    istream.read(signature);
	    for ( int i = 0 ; i < IMG_FILE_SIGNATURE.length ; i++ ) {
		if ( signature[i] != IMG_FILE_SIGNATURE[i] )
		    throw new IOException("Wrong file signature");
	    }
	    
	    // See what kind of compression is used.
	    byte[] used_compression = new byte[1];
	    istream.read(used_compression);
	    switch ( used_compression[0] ) {
		case GZIP_COMPRESSION:
		    gzstream = new GZIPInputStream(istream);
		    p = new ObjectInputStream(gzstream);
		    break;
		default:
		    p = new ObjectInputStream(istream);
		    break;
	    }
	    
	    // Read in the saved vector object.
	    readObject(p);
	    
	    // Close everything.
	    p.close();
	    if ( gzstream != null ) gzstream.close();
	    istream.close();
	}
	catch ( IOException e ) { System.err.println("IOException: " + e); return false; }
	catch ( ClassNotFoundException e ) { System.err.println("ClassNotFoundException: " + e); return false; }
	return true;
    }
    
    //=============================================================================
    // saveToFile
    //=============================================================================
    /**
     * Save the operations vector to a file. We just use standard Java
     * serialisation for this, except for pixmaps and special file/object
     * data. Return true if successful save. This method is synchronized
     * since we don't want anyone to mess around with the saved operations
     * before saving is finished.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    public synchronized boolean saveToFile(File f) {
	FileOutputStream	ostream;
	ObjectOutputStream	p;
	GZIPOutputStream	gzstream;
	byte[]	compression_byte = new byte[1];
	
	try {
	    // Open output stream and write the file signature.
	    ostream = new FileOutputStream(f);
	    ostream.write(IMG_FILE_SIGNATURE);
	    
	    // Indicate what kind of compression is used.
	    compression_byte[0] = (byte) GZIP_COMPRESSION;
	    ostream.write(compression_byte);
	    gzstream = new GZIPOutputStream(ostream);
	    
	    // Write out the recording.
	    p = new ObjectOutputStream(gzstream);
	    writeObject(p);
	    
	    // Flush and close everything. This is important
	    // especially for the GZIP stream.
	    p.flush(); p.close(); gzstream.close(); ostream.close();
	} catch ( IOException e ) { return false; }
	return true;
    }
    
    //--------------------------------------------------------------------------------------
    // Private methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // writeObject {SERIALIZABLE METHOD}
    //=============================================================================
    /**
     * Override the default serialization method so that we can treat
     * non-serializable objects (images for instance) and the file signature.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    private void writeObject(ObjectOutputStream out)
    throws IOException {
	// Write out the image objects first, which makes it easier to read
	// everything back afterwards.
	out.writeObject(imageVector);
	
	// Write out the operations object.
	out.writeObject(recordingVector);
    }
    
    //=============================================================================
    // readObject {SERIALIZABLE METHOD}
    //=============================================================================
    /**
     * Restore initial state from file.
     *
     * @author Kary FR&Auml;MLING 12/4/1998
     */
    //=============================================================================
    private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException {
	// Read in the image vector.
	imageVector = (ImageSerializerVector) in.readObject();
	
	// Read in the operations vector.
	recordingVector = (Vector) in.readObject();
    }
    
    //=============================================================================
    // getOpVector
    //=============================================================================
    /**
     * Copy the objects related to the operation in opVector at index opObjectIndex
     * into the opObjects vector, which is automatically emptied before. Return
     * the index of the next operation object. The end of operators is indicated
     * by an empty opObjects Vector.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //=============================================================================
    private int getOpVector(Vector opVector, int opObjectIndex, Vector opObjects)
    throws InvalidObjectException {
	GraphicsOperation	op = new GraphicsOperation(GraphicsOperation.NOP);
	
	// Empty the result vector.
	opObjects.removeAllElements();
	
	// Verify that the first object is indeed an operation object. Throw an
	// expection if not.
	if ( !(opVector.elementAt(opObjectIndex) instanceof GraphicsOperation) ) {
	    throw new InvalidObjectException("Not valid GraphicsOperation object");
	}
	
	// Copy it and the other objects into opObjects.
	opObjects.addElement(opVector.elementAt(opObjectIndex));
	opObjectIndex++;
	while ( opObjectIndex < opVector.size() &&
	!(opVector.elementAt(opObjectIndex) instanceof GraphicsOperation) ) {
	    
	    opObjects.addElement(opVector.elementAt(opObjectIndex));
	    opObjectIndex++;
	}
	
	// Return index to next operation.
	return opObjectIndex;
    }
    
    //=============================================================================
    // getNextObjectOfType
    //=============================================================================
    /**
     * Return all the objects of the class "searchClass" in the "objects" Vector.
     * The search begins from "startIndex" in "objects".
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //=============================================================================
    private Vector getObjectsOfType(Vector objects, int startIndex, Class searchClass)
    throws InvalidObjectException {
	Vector	objs = new Vector();
	
	for ( int i = startIndex ; i < objects.size() ; i++ ) {
	    if ( objects.elementAt(i).getClass() == searchClass )
		objs.addElement(objects.elementAt(i));
	}
	return objs;
    }
    
    //=============================================================================
    // getStoredImage
    //=============================================================================
    /**
     * See if "img" is already used in this recording and return its' index
     * in the stored image array. If it is not stored yet, it is added to
     * the vector.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //=============================================================================
    private int getStoredImage(Image img) {
	// Loop to find it instead of "indexOf" because "equals" is probably
	// not applicable to Image objects.
	for ( int i = 0 ; i < imageVector.size() ; i++ ) {
	    if ( img == ((ImageSerializer) imageVector.elementAt(i)).getImage() ) return i;
	}
	
	// Not found, add the image.
	imageVector.addElement(new ImageSerializer(img));
	return imageVector.size() - 1;
    }
}

