/*****************************************************************************
 *
 *                             TargaImageSaver.java
 *
 * Class for saving an Image object to file using the TrueVision Targa
 * format. Only 24-bit uncompressed format supported.
 *
 * Java source created by Kary FRAMLING 12/2/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ColorModel;
import java.util.Hashtable;
import java.awt.Toolkit;
import java.io.Serializable;

/**
 * Class for saving an Image object to file using the TrueVision Targa
 * format. Only 24-bit uncompressed format supported.
 *
 * @author Kary FR&Auml;MLING 12/2/1998
 */
public class TargaImageSaver extends ImageSaverInterface
implements Serializable, ImageConsumer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static String	FORMAT_CODE = "Targa";
    public static String	FORMAT_COMPLETE_NAME = "TrueVision Targa";
    public static String	FORMAT_EXTENSION = "tga";
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private static int		HEADER_SIZE = 18;
    private static int		IMAGE_ID_OFFSET = 18;
    private static int		PIXEL_DATA_SIZE = 24;
    private static int		NBR_BITS_IN_BYTE = 8;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private FileOutputStream	writeFileHandle;
    private int		saveStatus;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // main
    //=============================================================================
    /**
     * Main for testing this class.
     *
     * @author Kary FR&Auml;MLING 12/2/1998
     */
    //=============================================================================
    public static void main(String argv[]) {
	File		current_dir;
	Image		img;
	String		current_path, img_path;
	TargaImageSaver	is;
	
	// See that we have an image name.
	if ( argv.length < 2 ) {
	    System.out.println("Usage: TargaImageSaver <source image file> <dest image file>");
	    System.exit(1);
	}
	
	// Get the launch directory of the application.
	current_dir = new File(".");
	current_dir = new File(current_dir.getAbsolutePath());
	current_path = current_dir.getParent();
	
	// We use an image relative to our launch directory.
	img_path = current_path + File.separator + argv[0];
	img = Toolkit.getDefaultToolkit().getImage(img_path);
	is = new TargaImageSaver();
	is.setSaveImage(img);
	is.setSavePath(argv[1]);
	is.saveIt();
	
	// Wait until saving is ready.
	while ( (is.checkSave() & ( ImageObserver.ALLBITS | ImageObserver.ABORT | ImageObserver.ERROR )) == 0 ) {
	    System.out.println("Waiting for save to finish!");
	    try { Thread.sleep(100); }
	    catch ( InterruptedException e ) { System.exit(1); }
	}
	System.exit(0);
    }
    
    //=============================================================================
    /**
     * ImageSaverInterface method implementations.
     *
     * @author Kary FR&Auml;MLING 12/2/1998.
     */
    //=============================================================================
    public String getFormatCode() { return FORMAT_CODE; }
    public String getFormatString() { return FORMAT_COMPLETE_NAME; }
    public String getFormatExtension() { return FORMAT_EXTENSION; }
    
    //=============================================================================
    // saveIt
    //=============================================================================
    /**
     * Save the image.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public boolean saveIt() {
	ImageProducer	ip;
	
	// Verify that we have an image.
	if ( saveImage == null ) return false;
	
	// Open the file.
	try {
	    writeFileHandle = new FileOutputStream(savePath);
	} catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Return straight away if we couldn't get a handle.
	if ( writeFileHandle == null ) return false;
	
	// Set us up as image consumer and start producing.
	ip = saveImage.getSource();
	if ( ip == null ) return false;
	ip.startProduction(this);
	
	// Nothing more to do, just get data and close file at the end.
	return true;
    }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return saveStatus;
    }
    
    //=============================================================================
    /**
     * ImageConsumer method implementations.
     *
     * @author Kary FR&Auml;MLING 12/2/1998.
     */
    //=============================================================================
    public void setProperties(Hashtable props) {
	saveStatus |= ImageObserver.PROPERTIES;
    }
    
    public void setHints(int hintflags) {}
    public void setColorModel(ColorModel model) {}
    
    //=============================================================================
    // setDimensions
    //=============================================================================
    /**
     * Override the source image's dimensions and pass the dimensions
     * of the rectangular cropped region to the ImageConsumer.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void setDimensions(int w, int h) {
	byte[]	img_bytes;
	
	// Write the header information.
	// Create a pixel and a byte array that are big enough.
	img_bytes = new byte[HEADER_SIZE];
	
	// Fill out header information.
	img_bytes[0] = 0;
	img_bytes[1] = 0;
	img_bytes[2] = 2;
	img_bytes[3] = 0;
	img_bytes[4] = 0;
	img_bytes[5] = 0;
	img_bytes[6] = 0;
	img_bytes[7] = 0;
	img_bytes[8] = 0;
	img_bytes[9] = 0;
	img_bytes[10] = 0;
	img_bytes[11] = 0;
	img_bytes[12] = (byte) (w & 0xFF);
	img_bytes[13] = (byte) ((w >> 8) & 0xFF);
	img_bytes[14] = (byte) (h & 0xFF);
	img_bytes[15] = (byte) ((h >> 8) & 0xFF);
	img_bytes[16] = (byte) PIXEL_DATA_SIZE;
	img_bytes[17] = 0x20;
	
	// Write header information onto disk.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.write(img_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.WIDTH | ImageObserver.HEIGHT;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Override the source image's setPixels method so that we can replace all
     * non-transparent pixels with the given fill color or pattern. All transparent
     * pixels are replaced by the background color.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, byte pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb;
	byte[]	rgb_buf;
	int		pixel_byte_nbr;
	
	// One byte per pixel;
	pixel_byte_nbr = PIXEL_DATA_SIZE/NBR_BITS_IN_BYTE;
	
	// Create RGB buffer of same size as pixels.
	rgb_buf = new byte[w*h*pixel_byte_nbr];
	
	// Fill in pixel RGB data.
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		// Funny, we have to mask out the last byte for this to work correctly.
		rgb = model.getRGB(pixels[off + i*scansize + j]&0xFF);
		for ( pix_byte_ind = 0 ; pix_byte_ind < pixel_byte_nbr ; pix_byte_ind++ ) {
		    rgb_buf[(i*w + j)*pixel_byte_nbr + pix_byte_ind] =
		    (byte) ((rgb >> (pix_byte_ind*NBR_BITS_IN_BYTE)) & 0xFF);
		}
	    }
	}
	
	// Write header information onto disk.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.write(rgb_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Override the source image's setPixels method so that we can replace all
     * non-transparent pixels with the given fill color or pattern. All transparent
     * pixels are replaced by the background color.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, int pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb;
	byte[]	rgb_buf;
	int		pixel_byte_nbr;
	
	pixel_byte_nbr = PIXEL_DATA_SIZE/NBR_BITS_IN_BYTE;
	
	// Create RGB buffer of same size as pixels.
	rgb_buf = new byte[w*h*pixel_byte_nbr];
	
	// Fill in pixel RGB data.
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		rgb = model.getRGB(pixels[off + i*scansize + j]);
		for ( pix_byte_ind = 0 ; pix_byte_ind < pixel_byte_nbr ; pix_byte_ind++ ) {
		    rgb_buf[(i*w + j)*pixel_byte_nbr + pix_byte_ind] =
		    (byte) ((rgb >> (pix_byte_ind*NBR_BITS_IN_BYTE)) & 0xFF);
		}
	    }
	}
	
	// Write pixels onto disk.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.write(rgb_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // imageComplete
    //=============================================================================
    /**
     * Get imageComplete message so that we can close the output file.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void imageComplete(int status) {
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.close(); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.ALLBITS;
    }
    
}

