/*****************************************************************************
 *
 *                              TIFFImageSaver.java
 *
 * Class for saving an image in the TIFF format. Only uncompressed
 * supported for the moment.
 *
 * Java source created by Kary FRAMLING 12/2/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ColorModel;
import java.util.Hashtable;
import java.awt.Toolkit;
import java.io.Serializable;

/**
 * Class for saving an image in the TIFF format. Only uncompressed
 * supported for the moment.
 *
 * @author Kary FR&Auml;MLING 12/2/1998
 */
public class TIFFImageSaver extends ImageSaverInterface
implements Serializable, ImageConsumer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final String	FORMAT_CODE = "TIFF";
    public static final String	FORMAT_COMPLETE_NAME = "TIFF";
    public static final String	FORMAT_EXTENSION = "tif";
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final int		HEADER_SIZE = 8;
    private final int		IFD_FIELD_SIZE = 12;
    private final int		OFFSET_BYTE_SIZE = 4;
    private final int		NEXT_IFD_OFFSET_SIZE = OFFSET_BYTE_SIZE;
    private final int		BITS_PER_SAMPLE_SIZE = 6;
    private final int		RESOLUTION_DATA_SIZE = 8;
    private final int		BYTES_PER_PIXEL = 3;
    private final int		NBR_BITS_IN_BYTE = 8;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private FileOutputStream	writeFileHandle;
    private int		saveStatus;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    /**
     * ImageSaverInterface method implementations.
     *
     * @author Kary FR&Auml;MLING 12/2/1998.
     */
    //=============================================================================
    public String getFormatCode() { return FORMAT_CODE; }
    public String getFormatString() { return FORMAT_COMPLETE_NAME; }
    public String getFormatExtension() { return FORMAT_EXTENSION; }
    
    //=============================================================================
    // saveIt
    //=============================================================================
    /**
     * Save the image.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public boolean saveIt() {
	ImageProducer	ip;
	
	// Verify that we have an image.
	if ( saveImage == null ) return false;
	
	// No status information yet.
	saveStatus = 0;
	
	// Open the file.
	try {
	    writeFileHandle = new FileOutputStream(savePath);
	} catch ( IOException e ) { System.out.println("IOException occurred opening FileOutputStream : " + e); }
	
	// Return straight away if we couldn't get a handle.
	if ( writeFileHandle == null ) return false;
	
	// Set us up as image consumer and start producing.
	ip = saveImage.getSource();
	if ( ip == null ) return false;
	ip.startProduction(this);
	
	// Nothing more to do, just get data and close file at the end.
	return true;
    }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return saveStatus;
    }
    
    //=============================================================================
    /**
     * ImageConsumer method implementations.
     *
     * @author Kary FR&Auml;MLING 12/2/1998.
     */
    //=============================================================================
    public void setProperties(Hashtable props) {
	saveStatus |= ImageObserver.PROPERTIES;
    }
    
    public void setHints(int hintflags) {}
    public void setColorModel(ColorModel model) {}
    
    //=============================================================================
    // setDimensions
    //=============================================================================
    /**
     * Override the source image's dimensions and pass the dimensions
     * of the rectangular cropped region to the ImageConsumer.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void setDimensions(int w, int h) {
	
	int		strip_cnt, off;
	byte[]	strip_off_bytes, strip_cnt_bytes;
	
	// Initialize variables and calculate offsets.
	int nbr_IFD_fields = 13;				// Number of IFD fields.
	int bits_per_sample_offset = 			// Offset of BitsPerSample info.
	HEADER_SIZE + 2 + nbr_IFD_fields*IFD_FIELD_SIZE + NEXT_IFD_OFFSET_SIZE;
	int x_res_offset = 						// Offset of X resolution info.
	bits_per_sample_offset + BITS_PER_SAMPLE_SIZE;
	int y_res_offset = 						// Offset of Y resolution info.
	x_res_offset + RESOLUTION_DATA_SIZE;
	int strip_offsets_offset = 				// Offset of first strip offset.
	y_res_offset + RESOLUTION_DATA_SIZE;
	int rows_per_strip = 2;					// Calculate strip size as close to 8Kbytes as possible.
	for ( ; rows_per_strip*w*BYTES_PER_PIXEL <= 8192 ; rows_per_strip++ ) ;
	rows_per_strip--;
	int nbr_of_strips = 					// Number of strips.
	(h + rows_per_strip - 1)/rows_per_strip;
	int default_strip_byte_count = 			// Number of bytes/strip.
	rows_per_strip*w*BYTES_PER_PIXEL;
	int strip_byte_count_offsets = 			// Offset of strip byte counts.
	strip_offsets_offset + nbr_of_strips*OFFSET_BYTE_SIZE;
	int image_data_offset = 				// Offset of image data.
	strip_byte_count_offsets + nbr_of_strips*OFFSET_BYTE_SIZE;
	
	// Write the header information.
	// Create a pixel and a byte array that are big enough.
	byte[] header_bytes = new byte[HEADER_SIZE + 2];
	
	// Fill out header information.
	header_bytes[0] = 0x4D;	// Big-endian.
	header_bytes[1] = 0x4D;
	header_bytes[2] = 0;		// Identify as TIFF.
	header_bytes[3] = 42;
	header_bytes[4] = 0;		// Byte offset to first Image File Directory.
	header_bytes[5] = 0;
	header_bytes[6] = 0;
	header_bytes[7] = 8;
	header_bytes[8] = 0;		// All 12 required fields for RGB Full Color Images.
	header_bytes[9] = (byte) nbr_IFD_fields;
	
	// Write header information onto disk.
	try { writeFileHandle.write(header_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// NewSubFileType entry.
	byte[] NSFT_bytes = {00,(byte)0xFE, 00,04, 00,00,00,01, 00,00,00,00};
	try { writeFileHandle.write(NSFT_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// ImageWidth entry.
	byte[] width_bytes = {0x01,00, 00,04, 00,00,00,01, 00,00,(byte) ((w >> 8) & 0xFF),(byte) (w & 0xFF)};
	try { writeFileHandle.write(width_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// ImageHeight entry.
	byte[] height_bytes = {01,01, 00,04, 00,00,00,01, 00,00,(byte) ((h >> 8) & 0xFF),(byte) (h & 0xFF)};
	try { writeFileHandle.write(height_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// BitsPerSample entry. This is an offset to the true data (8,8,8).
	byte[] bps_bytes = {1,2, 0,3, 0,0,0,3, 0,0,
	(byte) ((bits_per_sample_offset >> 8) & 0xFF),
	(byte) (bits_per_sample_offset & 0xFF)
	};
	try { writeFileHandle.write(bps_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Compression entry.
	byte[] compression_bytes = {01,03, 00,03, 00,00,00,01, 00,01, 00,00};
	try { writeFileHandle.write(compression_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// PhotometricInterpration entry.
	byte[] PI_bytes = {01,06, 00,03, 00,00,00,01, 00,02, 00,00};
	try { writeFileHandle.write(PI_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// StripOffsets entry. Offset to the strip offsets entry.
	byte[] SO_bytes = {
	    1,(byte)0x11, 0,4,
	    (byte) ((nbr_of_strips >> 24) & 0xFF),
	    (byte) ((nbr_of_strips >> 16) & 0xFF),
	    (byte) ((nbr_of_strips >> 8) & 0xFF),
	    (byte) (nbr_of_strips & 0xFF),
	    0,0,
	    (byte) ((strip_offsets_offset >> 8) & 0xFF),
	    (byte) (strip_offsets_offset & 0xFF)
	};
	try { writeFileHandle.write(SO_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// SamplesPerPixel entry.
	byte[] SPP_bytes = {01,(byte)0x15, 00,03, 00,00,00,01, 00,03, 00,00};
	try { writeFileHandle.write(SPP_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// RowsPerStrip entry: Optimized to be as close to 8Kbytes as possible.
	byte[] RPS_bytes = {01,(byte)0x16, 00,04, 00,00,00,01,
	(byte) ((rows_per_strip >> 24) & 0xFF),
	(byte) ((rows_per_strip >> 16) & 0xFF),
	(byte) ((rows_per_strip >> 8) & 0xFF),
	(byte) (rows_per_strip & 0xFF),
	};
	try { writeFileHandle.write(RPS_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// StripByteCounts entry. Offset to the strip byte count entries.
	byte[] SBC_bytes = {
	    1,(byte)0x17, 0,4,
	    (byte) ((nbr_of_strips >> 24) & 0xFF),
	    (byte) ((nbr_of_strips >> 16) & 0xFF),
	    (byte) ((nbr_of_strips >> 8) & 0xFF),
	    (byte) (nbr_of_strips & 0xFF),
	    0,0,
	    (byte) ((strip_byte_count_offsets >> 8) & 0xFF),
	    (byte) (strip_byte_count_offsets & 0xFF)
	};
	try { writeFileHandle.write(SBC_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// XResolution entry: ratio of 1. Offset to the entry.
	byte[] XR_bytes = {1,(byte)0x1A, 0,5, 0,0,0,1, 0,0,
	(byte) ((x_res_offset >> 8) & 0xFF),
	(byte) (x_res_offset & 0xFF)
	};
	try { writeFileHandle.write(XR_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// YResolution entry: ratio of 1. Offset to the entry.
	byte[] YR_bytes = {01,(byte)0x1B, 00,05, 00,00,00,01, 0,0,
	(byte) ((y_res_offset >> 8) & 0xFF),
	(byte) (y_res_offset & 0xFF)
	};
	try { writeFileHandle.write(YR_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// ResolutionUnit entry.
	byte[] RU_bytes = {01,(byte)0x28, 00,03, 00,00,00,01, 00,01, 00,00};
	try { writeFileHandle.write(RU_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Next IFD offset (none).
	byte[] NIO_bytes = {00,00,00,00};
	try { writeFileHandle.write(NIO_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Bits per sample data.
	byte[] BPS_data_bytes = {0,8, 0,8, 0,8};
	try { writeFileHandle.write(BPS_data_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// X and Y resolution data.
	byte[] RES_data_bytes = {0,0,1,(byte)0x2C, 0,0,0,1};
	try { writeFileHandle.write(RES_data_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	try { writeFileHandle.write(RES_data_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Strip offsets.
	strip_off_bytes = new byte[OFFSET_BYTE_SIZE];
	for ( strip_cnt = 0, off = image_data_offset ; strip_cnt < nbr_of_strips ;
	strip_cnt++, off += default_strip_byte_count ) {
	    
	    strip_off_bytes[0] = (byte) ((off >> 24) & 0xFF);
	    strip_off_bytes[1] = (byte) ((off >> 16) & 0xFF);
	    strip_off_bytes[2] = (byte) ((off >> 8) & 0xFF);
	    strip_off_bytes[3] = (byte) (off & 0xFF);
	    try { writeFileHandle.write(strip_off_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Strip byte counts. They are all identical, except the last one.
	strip_off_bytes[0] = (byte) ((default_strip_byte_count >> 24) & 0xFF);
	strip_off_bytes[1] = (byte) ((default_strip_byte_count >> 16) & 0xFF);
	strip_off_bytes[2] = (byte) ((default_strip_byte_count >> 8) & 0xFF);
	strip_off_bytes[3] = (byte) (default_strip_byte_count & 0xFF);
	for ( strip_cnt = 0 ; strip_cnt < nbr_of_strips - 1 ; strip_cnt++ ) {
	    try { writeFileHandle.write(strip_off_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	int resting_bytes = (h - (nbr_of_strips - 1)*rows_per_strip)*w*BYTES_PER_PIXEL;
	strip_off_bytes[0] = (byte) ((resting_bytes >> 24) & 0xFF);
	strip_off_bytes[1] = (byte) ((resting_bytes >> 16) & 0xFF);
	strip_off_bytes[2] = (byte) ((resting_bytes >> 8) & 0xFF);
	strip_off_bytes[3] = (byte) (resting_bytes & 0xFF);
	try { writeFileHandle.write(strip_off_bytes); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Update save status.
	saveStatus |= ImageObserver.WIDTH | ImageObserver.HEIGHT;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 21/1/1998.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, byte pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb, rgb_offset;
	byte[]	rgb_buf;
	
	// Create RGB buffer of same size as pixels.
	rgb_buf = new byte[w*h*BYTES_PER_PIXEL];
	
	// Fill in pixel RGB data.
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		// Funny, we have to mask out the last byte for this to work correctly.
		rgb = model.getRGB(pixels[off + i*scansize + j]&0xFF);
		
		// Funny thing with TIFF files - they get colors in BGR order,
		// not in RGB. This is the reason for the strange offset calculation.
		rgb_offset = (i*w + j)*BYTES_PER_PIXEL + BYTES_PER_PIXEL - 1;
		for ( pix_byte_ind = 0 ; pix_byte_ind < BYTES_PER_PIXEL ; pix_byte_ind++ ) {
		    rgb_buf[rgb_offset - pix_byte_ind] =
		    (byte) ((rgb >> (pix_byte_ind*NBR_BITS_IN_BYTE)) & 0xFF);
		}
	    }
	}
	
	// Write header information onto disk.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.write(rgb_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 21/1/1998.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, int pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb, rgb_offset;
	byte[]	rgb_buf;
	
	// Create RGB buffer of same size as pixels.
	rgb_buf = new byte[w*h*BYTES_PER_PIXEL];
	
	// Fill in pixel RGB data.
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		rgb = model.getRGB(pixels[off + i*scansize + j]);
		
		// Funny thing with TIFF files - they get colors in BGR order,
		// not in RGB. This is the reason for the strange offset calculation.
		rgb_offset = (i*w + j)*BYTES_PER_PIXEL + BYTES_PER_PIXEL - 1;
		for ( pix_byte_ind = 0 ; pix_byte_ind < BYTES_PER_PIXEL ; pix_byte_ind++ ) {
		    rgb_buf[rgb_offset - pix_byte_ind] =
		    (byte) ((rgb >> (pix_byte_ind*NBR_BITS_IN_BYTE)) & 0xFF);
		}
	    }
	}
	
	// Write header information onto disk.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.write(rgb_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // imageComplete
    //=============================================================================
    /**
     * Get imageComplete message so that we can close the output file.
     * @see ImageConsumer.
     *
     * @author Kary FR&Auml;MLING 23/9/1997.
     */
    //=============================================================================
    public void imageComplete(int status) {
	// Close and clean up.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.close(); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.ALLBITS;
    }
    
}


