/*****************************************************************************
 *
 *                              PICTImageSaver.java
 *
 * Java source created by Kary FRAMLING 24/5/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ColorModel;
import java.util.Hashtable;
import java.awt.Toolkit;
import java.io.Serializable;

import fi.faidon.util.DataCompression;

/**
 * Class for saving an image in the Macintosh PICT format.
 * It is stored using the opcode "opDirectBitsRect", which directly
 * stores pixel RGB values packed by the packbits routines.
 *
 * @author Kary FR&Auml;MLING 24/5/1998
 */
public class PICTImageSaver extends ImageSaverInterface
implements Serializable, ImageConsumer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final String	FORMAT_CODE = "PICT";
    public static final String	FORMAT_COMPLETE_NAME = "Macintosh PICT";
    public static final String	FORMAT_EXTENSION = "pct";
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final int	SRC_COPY = 0;
    private final int	PICT_NULL_HEADER_SIZE = 512;
    private final int	IMAGE_RESOLUTION = 72;
    private final int	NBR_BYTES_IN_WORD = 2;
    private final int	NBR_BYTES_IN_LONG = 4;
    private final int	OP_CLIP_RGN = 0x01;
    private final int	OP_VERSION = 0x11;
    private final int	OP_DEF_HILITE = 0x1E;
    private final int	OP_V2_HEADER_OP = 0xC00;
    private final int	OP_DIRECT_BITS_RECT = 0x9A;
    private final int	OP_END_OF_PICTURE = 0xFF;
    private final int	OP_VERSION_2 = 0x2FF;
    private final int	EXT_VERSION_2_CODE = 0xFFFE;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private FileOutputStream	writeFileHandle;
    private int		width;
    private int		height;
    private int		rowBytes;
    private int		saveStatus;
    private int		byteCount;
    private byte[]	scanlineBytes;
    private byte[]	packedScanlineBytes;
    private int		scanWidthLeft;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    /**
     * ImageSaverInterface method implementations.
     */
    //=============================================================================
    public String getFormatCode() { return FORMAT_CODE; }
    public String getFormatString() { return FORMAT_COMPLETE_NAME; }
    public String getFormatExtension() { return FORMAT_EXTENSION; }
    
    //=============================================================================
    // saveIt
    //=============================================================================
    /**
     * Save the image.
     */
    //=============================================================================
    public boolean saveIt() {
	ImageProducer	ip;
	
	// Verify that we have an image.
	if ( saveImage == null ) return false;
	
	// No status information yet.
	saveStatus = 0;
	
	// Open the file.
	try {
	    writeFileHandle = new FileOutputStream(savePath);
	} catch ( IOException e ) { System.out.println("IOException occurred opening FileOutputStream : " + e); }
	
	// Return straight away if we couldn't get a handle.
	if ( writeFileHandle == null ) return false;
	
	// Set us up as image consumer and start producing.
	ip = saveImage.getSource();
	if ( ip == null ) return false;
	ip.startProduction(this);
	
	// Nothing more to do, just get data and close file at the end.
	return true;
    }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return saveStatus;
    }
    
    //=============================================================================
    /**
     * ImageConsumer method implementations.
     */
    //=============================================================================
    public void setProperties(Hashtable props) {
	saveStatus |= ImageObserver.PROPERTIES;
    }
    
    public void setHints(int hintflags) {}
    public void setColorModel(ColorModel model) {}
    
    //=============================================================================
    // setDimensions
    //=============================================================================
    /**
     */
    //=============================================================================
    public void setDimensions(int w, int h) {
	int		i;
	byte[]	buf;
	byte[]	byte_buf = new byte[1];
	byte[]	word_buf = new byte[NBR_BYTES_IN_WORD];
	byte[]	rect_buf = new byte[4*NBR_BYTES_IN_WORD];
	byte[]	long_buf = new byte[NBR_BYTES_IN_LONG];
	
	// Store width and height.
	width = w;
	height = h;
	
	// No bytes written yet.
	byteCount = 0;
	
	// Write the empty 512-byte header. We just set the first four
	// bytes to "PICT".
	buf = new byte[PICT_NULL_HEADER_SIZE];
	buf[0] = 'P'; buf[1] = 'I'; buf[2] = 'C'; buf[3] = 'T';
	try { writeFileHandle.write(buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += buf.length;
	
	// Write out the size. This is just zeros for new PICTs.
	for ( i = 0 ; i < word_buf.length ; i++ ) word_buf[i] = 0;
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Write out image frame.
	setIntAsBytes(0, rect_buf, 2, 4);
	setIntAsBytes(0, rect_buf, 0, 2);
	setIntAsBytes(w, rect_buf, 6, 8);
	setIntAsBytes(h, rect_buf, 4, 6);
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += rect_buf.length;
	
	// Write out version codes. We are producing extended version 2 images.
	setIntAsBytes(OP_VERSION, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	setIntAsBytes(OP_VERSION_2, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Version 2 HEADER_OP, extended version.
	setIntAsBytes(OP_V2_HEADER_OP, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	setIntAsBytes(EXT_VERSION_2_CODE, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Reserved byte.
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Image resolution, 72 dpi by default. This is a "fixed" value, whose meaning
	// I haven't caught completely yet. The second word always 0, in fact.
	setIntAsBytes(IMAGE_RESOLUTION, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(IMAGE_RESOLUTION, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += 4*word_buf.length;
	
	// Optimal source rectangle, same as frame rect for us.
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += rect_buf.length;
	
	// Reserved.
	setIntAsBytes(0, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// opDefHilite.
	setIntAsBytes(OP_DEF_HILITE, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Set the clip region.
	setIntAsBytes(OP_CLIP_RGN, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(10, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += 2*word_buf.length;
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += rect_buf.length;
	
	// Pixmap operation.
	setIntAsBytes(OP_DIRECT_BITS_RECT, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// PixMap pointer (always 0x000000FF);
	setIntAsBytes(0x000000FF, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// Write rowBytes, this is 4 times the width. Also set the
	// highest bit to 1 to indicate a PixMap.
	rowBytes = 4*w;
	setIntAsBytes(rowBytes, word_buf, 0, word_buf.length);
	word_buf[0] |= 0x80;
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Write bounds rectangle. Same as image bounds.
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += rect_buf.length;
	
	// PixMap record version number.
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Packing format. Always 4.
	setIntAsBytes(4, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Size of packed data. Always 0.
	setIntAsBytes(0, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// Pixmap resolution, 72 dpi by default.
	setIntAsBytes(IMAGE_RESOLUTION, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(IMAGE_RESOLUTION, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	setIntAsBytes(0, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += 4*word_buf.length;
	
	// Pixel type. 16 is allright for direct pixels.
	setIntAsBytes(16, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Pixel size.
	setIntAsBytes(32, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Pixel component count.
	setIntAsBytes(3, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Pixel component size.
	setIntAsBytes(8, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// planeBytes, ignored.
	setIntAsBytes(0, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// Handle to ColorTable record, there should be none for direct
	// bits so 0.
	setIntAsBytes(0, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// Reserved.
	setIntAsBytes(0, long_buf, 0, long_buf.length);
	try { writeFileHandle.write(long_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += long_buf.length;
	
	// Source and destination rectangles.
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	try { writeFileHandle.write(rect_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += 2*rect_buf.length;
	
	// Transfer mode.
	setIntAsBytes(SRC_COPY, word_buf, 0, word_buf.length);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	byteCount += word_buf.length;
	
	// Set up the buffers for storing scanline bytes and packed scanline bytes.
	// "scanWidthLeft" counts the number of bytes left for each scanline, since
	// bytes are packed and written on scanline basis.
	scanlineBytes = new byte[3*width];
	packedScanlineBytes = new byte[DataCompression.getPackBitsMaxDestBytes(scanlineBytes.length)];
	scanWidthLeft = width;
	
	// That's it. Now just pack the pixels into the file.
	
	// Update save status.
	saveStatus |= ImageObserver.WIDTH | ImageObserver.HEIGHT;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data. For this to work correctly,
     * pixels should be delivered in topdownleftright order with complete
     * scanlines. If we have several lines, the lines should be complete scanlines,
     * otherwise the saving fails.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, byte pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb, packed_size;
	byte[]	byte_buf = new byte[1];
	byte[]	word_buf = new byte[2];
	
	// Fill the scanline buffer. We get problems if ever we have several
	// lines (h > 1) and (w < width). This should never be the case.
	for ( i = 0 ; i < h ; i++ ) {
	    // Reduce the counter of bytes left on the scanline.
	    scanWidthLeft -= w;
	    
	    // Treat the scanline.
	    for ( j = 0 ; j < w ; j++ ) {
		// Funny, we have to mask out the last byte for this to work correctly.
		rgb = model.getRGB(pixels[off + i*scansize + j]&0xFF);
		
		// Set red, green and blue components.
		scanlineBytes[x + j] = (byte) ((rgb>>16)&0xFF);
		scanlineBytes[x + width + j] = (byte) ((rgb>>8)&0xFF);
		scanlineBytes[x + 2*width + j] = (byte) (rgb&0xFF);
	    }
	    
	    // If we have a complete scanline, then pack it and write it out.
	    if ( scanWidthLeft == 0 ) {
		packed_size = DataCompression.packBits(scanlineBytes, packedScanlineBytes, scanlineBytes.length);
		if ( rowBytes > 250 ) {
		    setIntAsBytes(packed_size, word_buf, 0, word_buf.length);
		    try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		    byteCount += word_buf.length;
		}
		else {
		    setIntAsBytes(packed_size, byte_buf, 0, byte_buf.length);
		    try { writeFileHandle.write(byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		    byteCount += byte_buf.length;
		}
		try { writeFileHandle.write(packedScanlineBytes, 0, packed_size); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount += packed_size;
		scanWidthLeft = width;
	    }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, int pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb, packed_size;
	byte[]	byte_buf = new byte[1];
	byte[]	word_buf = new byte[2];
	
	// Fill the scanline buffer. We get problems if ever we have several
	// lines (h > 1) and (w < width). This should never be the case.
	for ( i = 0 ; i < h ; i++ ) {
	    // Reduce the counter of bytes left on the scanline.
	    scanWidthLeft -= w;
	    
	    // Treat the scanline.
	    for ( j = 0 ; j < w ; j++ ) {
		// Funny, we have to mask out the last byte for this to work correctly.
		rgb = model.getRGB(pixels[off + i*scansize + j]);
		
		// Set red, green and blue components.
		scanlineBytes[x + j] = (byte) ((rgb>>16)&0xFF);
		scanlineBytes[x + width + j] = (byte) ((rgb>>8)&0xFF);
		scanlineBytes[x + 2*width + j] = (byte) (rgb&0xFF);
	    }
	    
	    // If we have a complete scanline, then pack it and write it out.
	    if ( scanWidthLeft == 0 ) {
		packed_size = DataCompression.packBits(scanlineBytes, packedScanlineBytes, scanlineBytes.length);
		if ( rowBytes > 250 ) {
		    setIntAsBytes(packed_size, word_buf, 0, word_buf.length);
		    try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		    byteCount += word_buf.length;
		}
		else {
		    setIntAsBytes(packed_size, byte_buf, 0, byte_buf.length);
		    try { writeFileHandle.write(byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		    byteCount += byte_buf.length;
		}
		try { writeFileHandle.write(packedScanlineBytes, 0, packed_size); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount += packed_size;
		scanWidthLeft = width;
	    }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // imageComplete
    //=============================================================================
    /**
     * Get imageComplete message so that we can close the output file.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void imageComplete(int status) {
	byte[]	byte_buf = new byte[1];
	byte[]	word_buf = new byte[NBR_BYTES_IN_WORD];
	
	// Write out end opcode. Be sure to be word-aligned.
	if ( (byteCount & 1) > 0 ) {
	    byte_buf[0] = 0;
	    try { writeFileHandle.write(byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	setIntAsBytes(OP_END_OF_PICTURE, word_buf, 0, 2);
	try { writeFileHandle.write(word_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	
	// Close and clean up.
	if ( writeFileHandle != null ) {
	    try { writeFileHandle.close(); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.ALLBITS;
    }
    
    //--------------------------------------------------------------------------------------
    // Private methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // setIntAsBytes
    //=============================================================================
    /**
     * Convert the int value into a byte array that starts from index "startOff" and
     * end at index "endOff" - 1 of "buf".
     */
    //=============================================================================
    private void setIntAsBytes(int value, byte[] bytes, int startOff, int endOff) {
	int		i, shift_cnt;
	
	for ( i = endOff - 1, shift_cnt = 0 ; i >= startOff ; i--, shift_cnt += 8 ) {
	    bytes[i] = (byte) ((value>>shift_cnt)&0xFF);
	}
    }
    
}


