/*****************************************************************************
 *
 *                           JPEGPrepController.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.IOException;
import java.util.Vector;

/**
 * This file contains the compression preprocessing controller.
 * This controller manages the color conversion, downsampling,
 * and edge expansion steps.
 *
 * Most of the complexity here is associated with buffering input rows
 * as required by the downsampler.  See the comments at the head of
 * jcsample.c for the downsampler's needs.
 *
 * @see jcprep.c of the IJG JPEG library Jpeg-6a.
 * @author Kary FR&Auml;MLING
 */
public class JPEGPrepController {
    
/*
 * For the simple (no-context-row) case, we just need to buffer one
 * row group's worth of pixels for the downsampling step.  At the bottom of
 * the image, we pad to a full row group by replicating the last pixel row.
 * The downsampler's last output row is then replicated if needed to pad
 * out to a full iMCU row.
 *
 * When providing context rows, we must buffer three row groups' worth of
 * pixels.  Three row groups are physically allocated, but the row pointer
 * arrays are made five row groups high, with the extra pointers above and
 * below "wrapping around" to point to the last and first real row groups.
 * This allows the downsampler to access the proper context rows.
 * At the top and bottom of the image, we create dummy context rows by
 * copying the first or last real pixel row.  This copying could be avoided
 * by pointer hacking as is done in jdmainct.c, but it doesn't seem worth the
 * trouble on the compression side.
 *
 * At present, jcsample.c can request context rows only for smoothing.
 * In the future, we might also need context rows for CCIR601 sampling
 * or other more-complex downsampling procedures.  The code to support
 * context rows should be compiled only if needed.
 */
    // We do not support context rows for the moment. KF.
    //#ifdef INPUT_SMOOTHING_SUPPORTED
    //#define CONTEXT_ROWS_SUPPORTED
    //#endif
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    
    private final int	START_PASS_PREP = 0;
    
/*	JMETHOD(void, start_pass, (j_compress_ptr cinfo, J_BUF_MODE pass_mode));
	JMETHOD(void, pre_process_data, (j_compress_ptr cinfo,
			JSAMPARRAY input_buf,
			JDIMENSION *in_row_ctr,
			JDIMENSION in_rows_avail,
			JSAMPIMAGE output_buf,
			JDIMENSION *out_row_group_ctr,
			JDIMENSION out_row_groups_avail));
 */
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    /**
     *Downsampling input buffer. This buffer holds color-converted data
     * until we have enough to do a downsample step.
     */
    private Vector	colorBuf = new Vector(JPEGImageSaver.MAX_COMPONENTS);
    
    private int	rowsToGo;			// Counts rows remaining in source image.
    private int	nextBufRow;			// Index of next row to store in color_buf.
    private int	startPassMethod;
    
    //#ifdef CONTEXT_ROWS_SUPPORTED	// only needed for context case
    private int	thisRowGroup;		// starting row index of group to process
    private int nextBufStop;		// downsample when we reach this index
    //#endif
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Initialize preprocessing controller.
     *
     * See jinit_c_prep_controller in jcprep.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public JPEGPrepController(JPEGImageSaver cinfo, boolean needFullBuffer) throws IOException {
	int		ci;
	JPEGComponentInfo	comp;
	
	if ( needFullBuffer )		// safety check.
	    throw new IOException(JPEGError.JERR_BAD_BUFFER_MODE);
	
	startPassMethod = START_PASS_PREP;
	
	// Allocate the color conversion buffer.
	// We make the buffer wide enough to allow the downsampler to edge-expand
	// horizontally within the buffer, if it so chooses.
/*		if (cinfo->downsample->need_context_rows) {
			// Set up to provide context rows.
//		#ifdef CONTEXT_ROWS_SUPPORTED
//			prep->pub.pre_process_data = pre_process_context;
//			create_context_buffer(cinfo);
//		#else
			throw new IOException(JPEGError.JERR_NOT_COMPILED);
//		#endif
		}
		else {
 */			// No context, just make it tall enough for one row group.
	//			prep->pub.pre_process_data = pre_process_data;
	for ( ci = 0 ; ci < cinfo.numComponents ; ci++ ) {
	    comp = (JPEGComponentInfo) cinfo.compInfoVector.elementAt(ci);
	    colorBuf.addElement(new int
	    [cinfo.maxVsampFactor]
	    [(comp.widthInBlocks*JPEGImageSaver.DCTSIZE*cinfo.maxHsampFactor)/comp.hSampFactor]
	    );
	}
	//		}
    }
    
    //=============================================================================
    // startPass
    //=============================================================================
    /**
     * Replaces start_pass function pointer in C implementation.
     */
    //=============================================================================
    public void startPass(JPEGImageSaver cinfo, int passMode) throws IOException {
	startPassMethod = START_PASS_PREP;
	switch ( startPassMethod ) {
	    case START_PASS_PREP:
		startPassPrep(cinfo, passMode);
		break;
	    default:
		break;
	}
    }
    
    //=============================================================================
    // startPassPrep
    //=============================================================================
    /**
     * Initialize for a processing pass.
     *
     * See start_pass_prep in jcprep.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void startPassPrep(JPEGImageSaver cinfo, int passMode) throws IOException {
	if ( passMode != JPEGImageSaver.JBUF_PASS_THRU )
	    throw new IOException(JPEGError.JERR_BAD_BUFFER_MODE);
	
	// Initialize total-height counter for detecting bottom of image.
	rowsToGo = cinfo.height;
	
	// Mark the conversion buffer empty.
	nextBufRow = 0;
	//#ifdef CONTEXT_ROWS_SUPPORTED
	// Preset additional state variables for context mode.
	// These aren't used in non-context mode, so we needn't test which mode.
	thisRowGroup = 0;
	
	// Set next_buf_stop to stop after two row groups have been read in.
	nextBufStop = 2*cinfo.maxVsampFactor;
	//#endif
    }
    
    //=============================================================================
    // expandBottomEdge
    //=============================================================================
    /**
     * Expand an image vertically from height input_rows to height output_rows,
     * by duplicating the bottom row.
     *
     * See expand_bottom_edge in jcprep.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void expandBottomEdge(int[][] imageData, int numCols,
    int inputRows, int outputRows) throws IOException {
	int		row;
	
	for ( row = inputRows ; row < outputRows ; row++ ) {
	    JPEGImageSaver.jcopySampleRows(imageData, inputRows - 1, imageData, row,
	    1, numCols);
	}
    }
    
    //=============================================================================
    // preProcessData
    //=============================================================================
    /**
     * Process some data in the simple no-context case.
     *
     * Preprocessor output data is counted in "row groups".  A row group
     * is defined to be v_samp_factor sample rows of each component.
     * Downsampling will produce this much data from each max_v_samp_factor
     * input rows.
     *
     * Since Java doesn't use parameter passing by reference, we pass the two
     * counters to update in the array rowCtrs instead. The original in_row_ctr
     * is rowCtrs[0] and out_row_group_ctr is rowCtrs[1]. KF.
     *
     * See pre_process_data in jcprep.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void preProcessData(JPEGImageSaver cinfo, int[] inputBuf, int[] rowCtrs,
    int inRowsAvail, Vector outputBuf, int outRowGroupsAvail, int off,
    int scansize) throws IOException {
/*
METHODDEF(void)
pre_process_data (j_compress_ptr cinfo,
		  JSAMPARRAY input_buf, JDIMENSION *in_row_ctr,
		  JDIMENSION in_rows_avail,
		  JSAMPIMAGE output_buf, JDIMENSION *out_row_group_ctr,
		  JDIMENSION out_row_groups_avail)
 */
	int		numrows, ci;
	int		inrows;
	JPEGComponentInfo	comp;
	
	while ( rowCtrs[0] < inRowsAvail &&
	rowCtrs[1] < outRowGroupsAvail) {
	    
	    // Do color conversion to fill the conversion buffer.
	    inrows = inRowsAvail - rowCtrs[0];
	    numrows = cinfo.maxVsampFactor - nextBufRow;
	    numrows = Math.min(numrows, inrows);
	    cinfo.cconvert.colorConvert(cinfo, inputBuf,
	    colorBuf, nextBufRow,
	    numrows, off, scansize);
	    
	    off += scansize;
	    rowCtrs[0] += numrows;
	    nextBufRow += numrows;
	    rowsToGo -= numrows;
	    
	    // If at bottom of image, pad to fill the conversion buffer.
	    if ( rowsToGo == 0 && nextBufRow < cinfo.maxVsampFactor ) {
		for ( ci = 0 ; ci < cinfo.numComponents ; ci++ ) {
		    expandBottomEdge((int[][]) colorBuf.elementAt(ci), cinfo.width,
		    nextBufRow, cinfo.maxVsampFactor);
		}
		nextBufRow = cinfo.maxVsampFactor;
	    }
	    
	    // If we've filled the conversion buffer, empty it.
	    if ( nextBufRow == cinfo.maxVsampFactor ) {
		cinfo.downsample.downsample(cinfo, colorBuf, 0,
		outputBuf, rowCtrs[1]);
		nextBufRow = 0;
		rowCtrs[1]++;
	    }
	    // If at bottom of image, pad the output to a full iMCU height.
	    // Note we assume the caller is providing a one-iMCU-height output buffer!
	    if ( rowsToGo == 0 && rowCtrs[1] < outRowGroupsAvail ) {
		for ( ci = 0 ; ci < cinfo.numComponents ; ci++ ) {
		    comp = (JPEGComponentInfo) cinfo.compInfoVector.elementAt(ci);
/*					expand_bottom_edge(output_buf[ci],
							compptr->width_in_blocks * DCTSIZE,
							(int) (*out_row_group_ctr * compptr->v_samp_factor),
							(int) (out_row_groups_avail * compptr->v_samp_factor));
 */
		}
		rowCtrs[1] = outRowGroupsAvail;
		break;			// can exit outer loop without test
	    }
	}
    }
    
    //#ifdef CONTEXT_ROWS_SUPPORTED
    
/*
 * Process some data in the context case.
 */
/*
METHODDEF(void)
pre_process_context (j_compress_ptr cinfo,
		     JSAMPARRAY input_buf, JDIMENSION *in_row_ctr,
		     JDIMENSION in_rows_avail,
		     JSAMPIMAGE output_buf, JDIMENSION *out_row_group_ctr,
		     JDIMENSION out_row_groups_avail)
{
  my_prep_ptr prep = (my_prep_ptr) cinfo->prep;
  int numrows, ci;
  int buf_height = cinfo->max_v_samp_factor * 3;
  JDIMENSION inrows;
 
  while (*out_row_group_ctr < out_row_groups_avail) {
    if (*in_row_ctr < in_rows_avail) {
      // Do color conversion to fill the conversion buffer.
      inrows = in_rows_avail - *in_row_ctr;
      numrows = prep->next_buf_stop - prep->next_buf_row;
      numrows = (int) MIN((JDIMENSION) numrows, inrows);
      (*cinfo->cconvert->color_convert) (cinfo, input_buf + *in_row_ctr,
					 prep->color_buf,
					 (JDIMENSION) prep->next_buf_row,
					 numrows);
      // Pad at top of image, if first time through
      if (prep->rows_to_go == cinfo->image_height) {
	for (ci = 0; ci < cinfo->num_components; ci++) {
	  int row;
	  for (row = 1; row <= cinfo->max_v_samp_factor; row++) {
	    jcopy_sample_rows(prep->color_buf[ci], 0,
			      prep->color_buf[ci], -row,
			      1, cinfo->image_width);
	  }
	}
      }
 *in_row_ctr += numrows;
      prep->next_buf_row += numrows;
      prep->rows_to_go -= numrows;
    } else {
      // Return for more data, unless we are at the bottom of the image.
      if (prep->rows_to_go != 0)
	break;
      // When at bottom of image, pad to fill the conversion buffer.
      if (prep->next_buf_row < prep->next_buf_stop) {
	for (ci = 0; ci < cinfo->num_components; ci++) {
	  expand_bottom_edge(prep->color_buf[ci], cinfo->image_width,
			     prep->next_buf_row, prep->next_buf_stop);
	}
	prep->next_buf_row = prep->next_buf_stop;
      }
    }
    // If we've gotten enough data, downsample a row group.
    if (prep->next_buf_row == prep->next_buf_stop) {
      (*cinfo->downsample->downsample) (cinfo,
					prep->color_buf,
					(JDIMENSION) prep->this_row_group,
					output_buf, *out_row_group_ctr);
      (*out_row_group_ctr)++;
      // Advance pointers with wraparound as necessary.
      prep->this_row_group += cinfo->max_v_samp_factor;
      if (prep->this_row_group >= buf_height)
	prep->this_row_group = 0;
      if (prep->next_buf_row >= buf_height)
	prep->next_buf_row = 0;
      prep->next_buf_stop = prep->next_buf_row + cinfo->max_v_samp_factor;
    }
  }
}
 */
    
    
/*
 * Create the wrapped-around downsampling input buffer needed for context mode.
 */
/*
LOCAL(void)
create_context_buffer (j_compress_ptr cinfo)
{
  my_prep_ptr prep = (my_prep_ptr) cinfo->prep;
  int rgroup_height = cinfo->max_v_samp_factor;
  int ci, i;
  jpeg_component_info * compptr;
  JSAMPARRAY true_buffer, fake_buffer;
 
  // Grab enough space for fake row pointers for all the components;
  // we need five row groups' worth of pointers for each component.
  fake_buffer = (JSAMPARRAY)
    (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_IMAGE,
				(cinfo->num_components * 5 * rgroup_height) *
				SIZEOF(JSAMPROW));
 
  for (ci = 0, compptr = cinfo->comp_info; ci < cinfo->num_components;
       ci++, compptr++) {
    // Allocate the actual buffer space (3 row groups) for this component.
    // We make the buffer wide enough to allow the downsampler to edge-expand
    // horizontally within the buffer, if it so chooses.
    true_buffer = (*cinfo->mem->alloc_sarray)
      ((j_common_ptr) cinfo, JPOOL_IMAGE,
       (JDIMENSION) (((long) compptr->width_in_blocks * DCTSIZE *
		      cinfo->max_h_samp_factor) / compptr->h_samp_factor),
       (JDIMENSION) (3 * rgroup_height));
    // Copy true buffer row pointers into the middle of the fake row array
    MEMCOPY(fake_buffer + rgroup_height, true_buffer,
	    3 * rgroup_height * SIZEOF(JSAMPROW));
    // Fill in the above and below wraparound pointers
    for (i = 0; i < rgroup_height; i++) {
      fake_buffer[i] = true_buffer[2 * rgroup_height + i];
      fake_buffer[4 * rgroup_height + i] = true_buffer[i];
    }
    prep->color_buf[ci] = fake_buffer + rgroup_height;
    fake_buffer += 5 * rgroup_height; // point to space for next component
  }
}
 */
    //#endif // CONTEXT_ROWS_SUPPORTED
    
    
} // End of class.

