/*****************************************************************************
 *
 *                               JPEGMaster.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.IOException;

/**
 * see jcmaster.c of the Independent JPEG Group's software.
 *
 * This file contains master control logic for the JPEG compressor.
 * These routines are concerned with parameter validation, initial setup,
 * and inter-pass control (determining the number of passes and the work
 * to be done in each pass).
 *
 * @author Kary FR&Auml;MLING
 */
class JPEGMaster {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    
    //--------------------------------------------------------------------------------------
    // Public fields.
    //--------------------------------------------------------------------------------------
    //  JMETHOD(void, prepare_for_pass, (j_compress_ptr cinfo));
    //  JMETHOD(void, pass_startup, (j_compress_ptr cinfo));
    //  JMETHOD(void, finish_pass, (j_compress_ptr cinfo));
    
    /**
     * State variables made visible to other modules.
     */
    boolean callPassStartup;	// True if pass_startup must be called.
    boolean isLastPass;		// True during last pass.
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final int	MAIN_PASS = 0;
    private final int	HUFF_OPT_PASS = 1;
    private final int	OUTPUT_PASS = 2;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private int		passType;		// the type of the current pass.
    private int		passNumber;		// # of passes completed
    private int		totalPasses;	// total # of passes needed
    private int		scanNumber;		// current index in scan_info[]
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Initialize master compression control.
     *
     * See jinit_c_master_control in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public JPEGMaster(JPEGImageSaver cinfo, boolean transcodeOnly) throws IOException {
/*		master->pub.prepare_for_pass = prepare_for_pass;
		master->pub.pass_startup = pass_startup;
		master->pub.finish_pass = finish_pass_master;
		master->pub.is_last_pass = FALSE;
 */
	// Validate parameters, determine derived values
	initialSetup(cinfo);
	
	if ( cinfo.scanInfo != null ) {
	    //		#ifdef C_MULTISCAN_FILES_SUPPORTED
	    //			validate_script(cinfo);
	    //		#else
	    throw new IOException(JPEGError.JERR_NOT_COMPILED);
	    //		#endif
	}
	else {
	    cinfo.progressiveMode = false;
	    cinfo.numScans = 1;
	}
	
	if ( cinfo.progressiveMode)			//  TEMPORARY HACK ???
	    cinfo.optimizeCoding = true;	// assume default tables no good for progressive mode
	
	// Initialize my private state
	if ( transcodeOnly ) {
	    // No main pass in transcoding.
	    if ( cinfo.optimizeCoding )
		passType = HUFF_OPT_PASS;
	    else
		passType = OUTPUT_PASS;
	}
	else {
	    // For normal compression, first pass is always this type.
	    passType = MAIN_PASS;
	}
	scanNumber = 0;
	passNumber = 0;
	if ( cinfo.optimizeCoding )
	    totalPasses = cinfo.numScans*2;
	else
	    totalPasses = cinfo.numScans;
    }
    
    //=============================================================================
    // initialSetup
    //=============================================================================
    /**
     * Do computations that are needed before master selection phase.
     *
     * See initial_setup in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void initialSetup(JPEGImageSaver cinfo) throws IOException {
	int		ci;
	long	samplesperrow;
	int		jd_samplesperrow;
	JPEGComponentInfo	comp;
	
	// Sanity check on image dimensions
	if ( cinfo.height <= 0 || cinfo.width <= 0
	|| cinfo.numComponents <= 0 || cinfo.inputComponents <= 0 )
	    throw new IOException(JPEGError.JERR_EMPTY_IMAGE);
	
	// Make sure image isn't bigger than I can handle
	if ( cinfo.height > JPEGImageSaver.JPEG_MAX_DIMENSION ||cinfo.width > JPEGImageSaver.JPEG_MAX_DIMENSION )
	    throw new IOException(JPEGError.JERR_IMAGE_TOO_BIG);
	
	// Width of an input scanline must be representable as JDIMENSION.
	samplesperrow = (long) cinfo.width*(long) cinfo.inputComponents;
	jd_samplesperrow = (int) samplesperrow;
	if ( (long) jd_samplesperrow != samplesperrow )
	    throw new IOException(JPEGError.JERR_WIDTH_OVERFLOW);
	
	// For now, precision must match compiled-in value...
	if ( cinfo.dataPrecision != JPEGImageSaver.BITS_IN_JSAMPLE )
	    throw new IOException(JPEGError.JERR_BAD_PRECISION);
	
	// Check that number of components won't exceed internal array sizes
	if ( cinfo.numComponents > JPEGImageSaver.MAX_COMPONENTS )
	    throw new IOException(JPEGError.JERR_COMPONENT_COUNT);
	
	// Compute maximum sampling factors; check factor validity
	cinfo.maxHsampFactor = 1;
	cinfo.maxVsampFactor = 1;
	for ( ci = 0 ; ci < cinfo.numComponents ; ci++ ) {
	    comp = (JPEGComponentInfo) cinfo.compInfoVector.elementAt(ci);
	    if ( comp.hSampFactor <= 0 || comp.hSampFactor > JPEGImageSaver.MAX_SAMP_FACTOR ||
	    comp.vSampFactor <= 0 || comp.vSampFactor > JPEGImageSaver.MAX_SAMP_FACTOR )
		throw new IOException(JPEGError.JERR_BAD_SAMPLING);
	    
	    cinfo.maxHsampFactor = Math.max(cinfo.maxHsampFactor,
	    comp.hSampFactor);
	    cinfo.maxVsampFactor = Math.max(cinfo.maxVsampFactor,
	    comp.vSampFactor);
	}
	
	// Compute dimensions of components
	for ( ci = 0 ; ci < cinfo.numComponents ; ci++ ) {
	    comp = (JPEGComponentInfo) cinfo.compInfoVector.elementAt(ci);
	    
	    // Fill in the correct component_index value; don't rely on application
	    comp.componentIndex = ci + 1;	// Strange, corrected to + 1 by KF, but there's something strange here!
	    
	    // For compression, we never do DCT scaling.
	    comp.DCTscaledSize = JPEGImageSaver.DCTSIZE;
	    
	    // Size in DCT blocks. The division is done by jdiv_round_up of JQUANT1.C in the JPEG
	    // library,  but this operation should be the same.
	    comp.widthInBlocks = (int) Math.ceil(((long) cinfo.width*(long) comp.hSampFactor)/
	    (long) (cinfo.maxHsampFactor*JPEGImageSaver.DCTSIZE));
	    comp.heightInBlocks = (int)
	    Math.ceil(((long) cinfo.height*(long) comp.vSampFactor)/
	    (long) (cinfo.maxVsampFactor*JPEGImageSaver.DCTSIZE));
	    
	    // Size in samples
	    comp.downsampledWidth = (int)
	    Math.ceil(((long) cinfo.width*(long) comp.hSampFactor)/
	    (long) cinfo.maxHsampFactor);
	    comp.downsampledHeight = (int)
	    Math.ceil(((long) cinfo.height*(long) comp.vSampFactor)/
	    (long) cinfo.maxVsampFactor);
	    
	    // Mark component needed (this flag isn't actually used for compression)
	    comp.componentNeeded = true;
	}
	
	// Compute number of fully interleaved MCU rows (number of times that
	// main controller will call coefficient controller).
	cinfo.totaliMCURows = (int)
	Math.ceil((long) cinfo.height/
	(long) (cinfo.maxVsampFactor*JPEGImageSaver.DCTSIZE));
	
    }
    
    //#ifdef C_MULTISCAN_FILES_SUPPORTED
/*
LOCAL(void)
validate_script (j_compress_ptr cinfo)
 */
/* Verify that the scan script in cinfo->scan_info[] is valid; also
 * determine whether it uses progressive JPEG, and set cinfo->progressive_mode.
 */
/*
{
  const jpeg_scan_info * scanptr;
  int scanno, ncomps, ci, coefi, thisi;
  int Ss, Se, Ah, Al;
  boolean component_sent[JPEGImageSaver.MAX_COMPONENTS];
//#ifdef C_PROGRESSIVE_SUPPORTED
  int * last_bitpos_ptr;
  int last_bitpos[JPEGImageSaver.MAX_COMPONENTS][JPEGImageSaver.DCTSIZE2];
  // -1 until that coefficient has been seen; then last Al for it
//#endif
 
  if (cinfo->num_scans <= 0)
    ERREXIT1(cinfo, JERR_BAD_SCAN_SCRIPT, 0);
 
  // For sequential JPEG, all scans must have Ss=0, Se=JPEGImageSaver.DCTSIZE2-1;
  // for progressive JPEG, no scan can have this.
  scanptr = cinfo->scan_info;
  if (scanptr->Ss != 0 || scanptr->Se != JPEGImageSaver.DCTSIZE2-1) {
//#ifdef C_PROGRESSIVE_SUPPORTED
    cinfo->progressive_mode = TRUE;
    last_bitpos_ptr = & last_bitpos[0][0];
    for (ci = 0; ci < cinfo->num_components; ci++)
      for (coefi = 0; coefi < JPEGImageSaver.DCTSIZE2; coefi++)
 *last_bitpos_ptr++ = -1;
//#else
    ERREXIT(cinfo, JERR_NOT_COMPILED);
//#endif
  } else {
    cinfo->progressive_mode = FALSE;
    for (ci = 0; ci < cinfo->num_components; ci++)
      component_sent[ci] = FALSE;
  }
 
  for (scanno = 1; scanno <= cinfo->num_scans; scanptr++, scanno++) {
    // Validate component indexes
    ncomps = scanptr->comps_in_scan;
    if (ncomps <= 0 || ncomps > MAX_COMPS_IN_SCAN)
      ERREXIT2(cinfo, JERR_COMPONENT_COUNT, ncomps, MAX_COMPS_IN_SCAN);
    for (ci = 0; ci < ncomps; ci++) {
      thisi = scanptr->component_index[ci];
      if (thisi < 0 || thisi >= cinfo->num_components)
	ERREXIT1(cinfo, JERR_BAD_SCAN_SCRIPT, scanno);
      // Components must appear in SOF order within each scan
      if (ci > 0 && thisi <= scanptr->component_index[ci-1])
	ERREXIT1(cinfo, JERR_BAD_SCAN_SCRIPT, scanno);
    }
    // Validate progression parameters
    Ss = scanptr->Ss;
    Se = scanptr->Se;
    Ah = scanptr->Ah;
    Al = scanptr->Al;
    if (cinfo->progressive_mode) {
//#ifdef C_PROGRESSIVE_SUPPORTED
      if (Ss < 0 || Ss >= JPEGImageSaver.DCTSIZE2 || Se < Ss || Se >= JPEGImageSaver.DCTSIZE2 ||
	  Ah < 0 || Ah > 13 || Al < 0 || Al > 13)
	ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
      if (Ss == 0) {
	if (Se != 0)		// DC and AC together not OK
	  ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
      } else {
	if (ncomps != 1)	// AC scans must be for only one component
	  ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
      }
      for (ci = 0; ci < ncomps; ci++) {
	last_bitpos_ptr = & last_bitpos[scanptr->component_index[ci]][0];
	if (Ss != 0 && last_bitpos_ptr[0] < 0) // AC without prior DC scan
	  ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
	for (coefi = Ss; coefi <= Se; coefi++) {
	  if (last_bitpos_ptr[coefi] < 0) {
	    // first scan of this coefficient
	    if (Ah != 0)
	      ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
	  } else {
	    // not first scan
	    if (Ah != last_bitpos_ptr[coefi] || Al != Ah-1)
	      ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
	  }
	  last_bitpos_ptr[coefi] = Al;
	}
      }
//#endif
    } else {
      // For sequential JPEG, all progression parameters must be these:
      if (Ss != 0 || Se != JPEGImageSaver.DCTSIZE2-1 || Ah != 0 || Al != 0)
	ERREXIT1(cinfo, JERR_BAD_PROG_SCRIPT, scanno);
      // Make sure components are not sent twice
      for (ci = 0; ci < ncomps; ci++) {
	thisi = scanptr->component_index[ci];
	if (component_sent[thisi])
	  ERREXIT1(cinfo, JERR_BAD_SCAN_SCRIPT, scanno);
	component_sent[thisi] = TRUE;
      }
    }
  }
 
  // Now verify that everything got sent.
  if (cinfo->progressive_mode) {
//#ifdef C_PROGRESSIVE_SUPPORTED
    // For progressive mode, we only check that at least some DC data
    // got sent for each component; the spec does not require that all bits
    // of all coefficients be transmitted.  Would it be wiser to enforce
    // transmission of all coefficient bits??
    for (ci = 0; ci < cinfo->num_components; ci++) {
      if (last_bitpos[ci][0] < 0)
	ERREXIT(cinfo, JERR_MISSING_DATA);
    }
//#endif
  } else {
    for (ci = 0; ci < cinfo->num_components; ci++) {
      if (! component_sent[ci])
	ERREXIT(cinfo, JERR_MISSING_DATA);
    }
  }
}
 
//#endif // C_MULTISCAN_FILES_SUPPORTED
 */
    
    //=============================================================================
    // selectScanParameters
    //=============================================================================
    /**
     * Set up the scan parameters for the current scan.
     *
     * See select_scan_parameters in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void selectScanParameters(JPEGImageSaver cinfo) throws IOException {
	int		ci;
	
/*		//#ifdef C_MULTISCAN_FILES_SUPPORTED
		if (cinfo->scan_info != NULL) {
			// Prepare for current scan --- the script is already validated
			my_master_ptr master = (my_master_ptr) cinfo->master;
			const jpeg_scan_info * scanptr = cinfo->scan_info + master->scan_number;
 
			cinfo->comps_in_scan = scanptr->comps_in_scan;
			for (ci = 0; ci < scanptr->comps_in_scan; ci++) {
				cinfo->cur_comp_info[ci] =
						&cinfo->comp_info[scanptr->component_index[ci]];
			}
			cinfo->Ss = scanptr->Ss;
			cinfo->Se = scanptr->Se;
			cinfo->Ah = scanptr->Ah;
			cinfo->Al = scanptr->Al;
		}
		else
		//#endif
 */
	{
	    // Prepare for single sequential-JPEG scan containing all components
	    if ( cinfo.numComponents > JPEGImageSaver.MAX_COMPS_IN_SCAN )
		throw new IOException(JPEGError.JERR_COMPONENT_COUNT);
	    //				ERREXIT2(cinfo, JERR_COMPONENT_COUNT, cinfo->num_components,
	    //						MAX_COMPS_IN_SCAN);
	    cinfo.compsInScan = cinfo.numComponents;
	    for ( ci = 0 ; ci < cinfo.numComponents ; ci++) {
		cinfo.curCompInfo.setElementAt(cinfo.compInfoVector.elementAt(ci), ci);
	    }
	    
	    cinfo.Ss = 0;
	    cinfo.Se = JPEGImageSaver.DCTSIZE2 - 1;
	    cinfo.Ah = 0;
	    cinfo.Al = 0;
	}
    }
    
    //=============================================================================
    // perScanSetup
    //=============================================================================
    /**
     * Do computations that are needed before processing a JPEG scan.
     * cinfo->comps_in_scan and cinfo->cur_comp_info[] are already set.
     *
     * See per_scan_setup in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void perScanSetup(JPEGImageSaver cinfo) throws IOException {
	int		ci, mcublks, tmp, nominal;
	JPEGComponentInfo	comp;
	
	System.out.println("perScanSetup");
	if ( cinfo.compsInScan == 1 ) {
	    // Noninterleaved (single-component) scan.
	    comp = (JPEGComponentInfo) cinfo.curCompInfo.elementAt(0);
	    
	    // Overall image size in MCUs
	    cinfo.MCUsPerRow = comp.widthInBlocks;
	    cinfo.MCURowsInScan = comp.heightInBlocks;
	    
	    // For noninterleaved scan, always one block per MCU
	    comp.MCUwidth = 1;
	    comp.MCUheight = 1;
	    comp.MCUblocks = 1;
	    comp.MCUsampleWidth = JPEGImageSaver.DCTSIZE;
	    comp.lastColWidth = 1;
	    
	    // For noninterleaved scans, it is convenient to define last_row_height
	    // as the number of block rows present in the last iMCU row.
	    tmp = (int) (comp.heightInBlocks%comp.vSampFactor);
	    if ( tmp == 0 ) tmp = comp.vSampFactor;
	    comp.lastRowHeight = tmp;
	    
	    // Prepare array describing MCU composition
	    cinfo.blocksInMCU = 1;
	    cinfo.MCUmembership[0] = 0;
	}
	else {
	    // Interleaved (multi-component) scan
	    if ( cinfo.compsInScan <= 0 || cinfo.compsInScan > JPEGImageSaver.MAX_COMPS_IN_SCAN )
		throw new IOException(JPEGError.JERR_COMPONENT_COUNT);
	    //				ERREXIT2(cinfo, JERR_COMPONENT_COUNT, cinfo->comps_in_scan,
	    //						MAX_COMPS_IN_SCAN);
	    
	    // Overall image size in MCUs
	    cinfo.MCUsPerRow = (int)
	    Math.ceil(cinfo.width/(cinfo.maxHsampFactor*JPEGImageSaver.DCTSIZE));
	    cinfo.MCURowsInScan = (int)
	    Math.ceil(cinfo.height/(cinfo.maxVsampFactor*JPEGImageSaver.DCTSIZE));
	    
	    cinfo.blocksInMCU = 0;
	    
	    for (ci = 0 ; ci < cinfo.compsInScan ; ci++ ) {
		comp = (JPEGComponentInfo) cinfo.curCompInfo.elementAt(ci);
		
		// Sampling factors give # of blocks of component in each MCU.
		comp.MCUwidth = comp.hSampFactor;
		comp.MCUheight = comp.vSampFactor;
		comp.MCUblocks = comp.MCUwidth*comp.MCUheight;
		comp.MCUsampleWidth = comp.MCUwidth * JPEGImageSaver.DCTSIZE;
		
		// Figure number of non-dummy blocks in last MCU column & row.
		tmp = (int) (comp.widthInBlocks % comp.MCUwidth);
		if ( tmp == 0 ) tmp = comp.MCUwidth;
		comp.lastColWidth = tmp;
		tmp = (int) (comp.heightInBlocks % comp.MCUheight);
		if ( tmp == 0 ) tmp = comp.MCUheight;
		comp.lastRowHeight = tmp;
		
		// Prepare array describing MCU composition.
		mcublks = comp.MCUblocks;
		if ( cinfo.blocksInMCU + mcublks > JPEGImageSaver.C_MAX_BLOCKS_IN_MCU )
		    throw new IOException(JPEGError.JERR_BAD_MCU_SIZE);
		while ( mcublks-- > 0 ) {
		    cinfo.MCUmembership[cinfo.blocksInMCU++] = ci;
		}
	    }
	}
	
	// Convert restart specified in rows to actual MCU count.
	// Note that count must fit in 16 bits, so we provide limiting.
	if ( cinfo.restartInRows > 0 ) {
	    nominal = cinfo.restartInRows*cinfo.MCUsPerRow;
	    cinfo.restartInterval = (int) Math.min(nominal, 65535L);
	}
    }
    
    //=============================================================================
    // prepareForPass
    //=============================================================================
    /**
     * Per-pass setup.
     * This is called at the beginning of each pass.  We determine which modules
     * will be active during this pass and give them appropriate start_pass calls.
     * We also set is_last_pass to indicate whether any more passes will be
     * required.
     *
     * See prepare_for_pass in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void prepareForPass(JPEGImageSaver cinfo) throws IOException {
	switch ( passType ) {
	    case MAIN_PASS:
		System.out.println("MAIN_PASS");
		
		// Initial pass: will collect input data, and do either Huffman
		// optimization or data output for the first scan.
		selectScanParameters(cinfo);
		perScanSetup(cinfo);
		if ( !cinfo.rawDataIn ) {
		    cinfo.cconvert.startPass(cinfo);
		    cinfo.downsample.startPass(cinfo);
		    cinfo.prep.startPass(cinfo, JPEGImageSaver.JBUF_PASS_THRU);
		}
/*			(*cinfo->fdct->start_pass) (cinfo);
			(*cinfo->entropy->start_pass) (cinfo, cinfo->optimize_coding);
			(*cinfo->coef->start_pass) (cinfo,
					(master->total_passes > 1 ?
					JBUF_SAVE_AND_PASS : JBUF_PASS_THRU));
			(*cinfo->main->start_pass) (cinfo, JBUF_PASS_THRU);
 */
		if ( cinfo.optimizeCoding) {
		    // No immediate data output; postpone writing frame/scan headers
		    callPassStartup = false;
		} else {
		    // Will write frame/scan headers at first jpeg_write_scanlines call
		    callPassStartup = true;
		}
		break;
		
		//		#ifdef ENTROPY_OPT_SUPPORTED
	    case HUFF_OPT_PASS:
		// Do Huffman optimization for a scan after the first one.
		selectScanParameters(cinfo);
		perScanSetup(cinfo);
		if ( cinfo.Ss != 0 || cinfo.Ah == 0 || cinfo.arithCode ) {
		    //				(*cinfo->entropy->start_pass) (cinfo, TRUE);
		    //				(*cinfo->coef->start_pass) (cinfo, JBUF_CRANK_DEST);
		    callPassStartup = false;
		    break;
		}
		// Special case: Huffman DC refinement scans need no Huffman table
		// and therefore we can skip the optimization pass for them.
		passType = OUTPUT_PASS;
		passNumber++;
		// *FALLTHROUGH
		//		#endif
	    case OUTPUT_PASS:
		System.out.println("OUTPUT_PASS");
		// Do a data-output pass.
		// We need not repeat per-scan setup if prior optimization pass did it.
		if ( !cinfo.optimizeCoding) {
		    selectScanParameters(cinfo);
		    perScanSetup(cinfo);
		}
		//			(*cinfo->entropy->start_pass) (cinfo, FALSE);
		//			(*cinfo->coef->start_pass) (cinfo, JBUF_CRANK_DEST);
		
		// We emit frame/scan headers now
		if ( scanNumber == 0 )
		    cinfo.writeFrameHeader();
		cinfo.writeScanHeader();
		callPassStartup = false;
		break;
	    default:
		throw new IOException(JPEGError.JERR_NOT_COMPILED);
	}
	
	isLastPass = (passNumber == totalPasses - 1);
	
	// Set up progress monitor's pass info if present
/*		if (cinfo->progress != NULL) {
			cinfo->progress->completed_passes = master->pass_number;
			cinfo->progress->total_passes = master->total_passes;
		}
 */
    }
    
    //=============================================================================
    // passStartup
    //=============================================================================
    /**
     * Special start-of-pass hook.
     * This is called by jpeg_write_scanlines if call_pass_startup is TRUE.
     * In single-pass processing, we need this hook because we don't want to
     * write frame/scan headers during jpeg_start_compress; we want to let the
     * application write COM markers etc. between jpeg_start_compress and the
     * jpeg_write_scanlines loop.
     * In multi-pass processing, this routine is not used.
     *
     * See pass_startup in jcmaster.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void passStartup(JPEGImageSaver cinfo) throws IOException {
	callPassStartup = false;	// reset flag so call only once.
	
	cinfo.writeFrameHeader();
	cinfo.writeScanHeader();
    }
    
/*
 * Finish up at end of pass.
 */
/*
METHODDEF(void)
finish_pass_master (j_compress_ptr cinfo)
{
  my_master_ptr master = (my_master_ptr) cinfo->master;
 
  // The entropy coder always needs an end-of-pass call,
  // either to analyze statistics or to flush its output buffer.
  (*cinfo->entropy->finish_pass) (cinfo);
 
  // Update state for next pass
  switch (master->pass_type) {
  case main_pass:
    // next pass is either output of scan 0 (after optimization)
    // or output of scan 1 (if no optimization).
    master->pass_type = output_pass;
    if (! cinfo->optimize_coding)
      master->scan_number++;
    break;
  case huff_opt_pass:
    // next pass is always output of current scan
    master->pass_type = output_pass;
    break;
  case output_pass:
    // next pass is either optimization or output of next scan
    if (cinfo->optimize_coding)
      master->pass_type = huff_opt_pass;
    master->scan_number++;
    break;
  }
 
  master->pass_number++;
}
 */
    
} // End of class.

