/*****************************************************************************
 *
 *                           JPEGMainController.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.IOException;
import java.util.Vector;

/**
 * This file contains the main buffer controller for compression.
 * The main buffer lies between the pre-processor and the JPEG
 * compressor proper; it holds downsampled data in the JPEG colorspace.
 *
 * @see jcmainct.c of the IJG JPEG library Jpeg-6a.
 * @author Kary FR&Auml;MLING
 */

class JPEGMainController {
    
	/* Note: currently, there is no operating mode in which a full-image buffer
	 * is needed at this step.  If there were, that mode could not be used with
	 * "raw data" input, since this module is bypassed in that case.  However,
	 * we've left the code here for possible use in special applications.
	 */
    //#undef FULL_MAIN_BUFFER_SUPPORTED
    
    //--------------------------------------------------------------------------------------
    // Public fields.
    //--------------------------------------------------------------------------------------
/*
struct jpeg_c_main_controller {
  JMETHOD(void, start_pass, (j_compress_ptr cinfo, J_BUF_MODE pass_mode));
  JMETHOD(void, process_data, (j_compress_ptr cinfo,
			       JSAMPARRAY input_buf, JDIMENSION *in_row_ctr,
			       JDIMENSION in_rows_avail));
};
 */
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    
    private final int	START_PASS_MAIN = 0;
    private final int	PROCESS_DATA_SIMPLE_MAIN = 0;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    
    private int		curiMCURow;		// Number of current iMCU row.
    private int		rowgroupCtr;	// Counts row groups received in iMCU row.
    private boolean suspended;		// Remember if we suspended output.
    private int		passMode;		// Current operating mode.
    private int		startPassMethod;	// Replaces function pointers.
    private int		processDataMethod;	// Replaces process_data function pointer.
    
    // If using just a strip buffer, this points to the entire set of buffers
    // (we allocate one for each component).  In the full-image case, this
    // points to the currently accessible strips of the virtual arrays.
    private Vector	buffer = new Vector(JPEGImageSaver.MAX_COMPONENTS);
/*
#ifdef FULL_MAIN_BUFFER_SUPPORTED
	// If using full-image storage, this array holds pointers to virtual-array
	// control blocks for each component.  Unused if not full-image storage.
	jvirt_sarray_ptr whole_image[MAX_COMPONENTS];
#endif
 */
    
    /* Forward declarations */
/*
METHODDEF(void) process_data_simple_main
	JPP((j_compress_ptr cinfo, JSAMPARRAY input_buf,
	     JDIMENSION *in_row_ctr, JDIMENSION in_rows_avail));
#ifdef FULL_MAIN_BUFFER_SUPPORTED
METHODDEF(void) process_data_buffer_main
	JPP((j_compress_ptr cinfo, JSAMPARRAY input_buf,
	     JDIMENSION *in_row_ctr, JDIMENSION in_rows_avail));
#endif
 */
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Initialize main buffer controller.
     *
     * See jinit_c_main_controller in jcmainct.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public JPEGMainController(JPEGImageSaver cinfo, boolean needFullBuffer) throws IOException {
	int		ci;
	JPEGComponentInfo	comp;
	
	startPassMethod = START_PASS_MAIN;
	
	// We don't need to create a buffer in raw-data mode.
	if (cinfo.rawDataIn)
	    return;
	
	// Create the buffer.  It holds downsampled data, so each component
	// may be of a different size.
	if (needFullBuffer) {
	    // Never supported. KF.
/*
//#ifdef FULL_MAIN_BUFFER_SUPPORTED
			// Allocate a full-image virtual array for each component
			// Note we pad the bottom to a multiple of the iMCU height
			for (ci = 0, compptr = cinfo->comp_info; ci < cinfo->num_components;
					ci++, compptr++) {
				main->whole_image[ci] = (*cinfo->mem->request_virt_sarray)
						((j_common_ptr) cinfo, JPOOL_IMAGE, FALSE,
						compptr->width_in_blocks * DCTSIZE,
						(JDIMENSION) jround_up((long) compptr->height_in_blocks,
						(long) compptr->v_samp_factor) * DCTSIZE,
						(JDIMENSION) (compptr->v_samp_factor * DCTSIZE));
			}
//		#else */
	    throw new IOException(JPEGError.JERR_BAD_BUFFER_MODE);
	    /*		#endif */
	}
	else {
/*
#ifdef FULL_MAIN_BUFFER_SUPPORTED
			main->whole_image[0] = NULL; // Flag for no virtual arrays.
#endif
 */
	    // Allocate a strip buffer for each component.
	    for ( ci = 0 ; ci < cinfo.numComponents ; ci++) {
		comp = (JPEGComponentInfo) cinfo.compInfoVector.elementAt(ci);
		buffer.addElement(new int
		[(int) (comp.vSampFactor*JPEGImageSaver.DCTSIZE)]
		[comp.widthInBlocks*JPEGImageSaver.DCTSIZE]
		);
	    }
	}
    }
    
    //=============================================================================
    // startPass
    //=============================================================================
    /**
     * Initialize for a processing pass.
     *
     * See jinit_c_main_controller in jcmainct.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void startPass(JPEGImageSaver cinfo, int newPassMode) throws IOException {
	if ( startPassMethod == START_PASS_MAIN ) {
	    startPassMain(cinfo, newPassMode);
	}
	else {
	    throw new IOException(JPEGError.JERR_BAD_MAIN_CT_PASS);
	}
    }
    
    //=============================================================================
    // startPassMain
    //=============================================================================
    /**
     * Initialize for a processing pass.
     *
     * See jinit_c_main_controller in jcmainct.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void startPassMain(JPEGImageSaver cinfo, int newPassMode) throws IOException {
	// Do nothing in raw-data mode.
	if ( cinfo.rawDataIn )
	    return;
	
	// Initialize counters.
	curiMCURow = 0;
	rowgroupCtr = 0;
	suspended = false;
	passMode = newPassMode;	// Save mode for use by process_data.
	
	switch (passMode) {
	    case JPEGImageSaver.JBUF_PASS_THRU:
/*
#ifdef FULL_MAIN_BUFFER_SUPPORTED
			if (main->whole_image[0] != NULL)
				ERREXIT(cinfo, JERR_BAD_BUFFER_MODE);
#endif
 */
		processDataMethod = PROCESS_DATA_SIMPLE_MAIN;
		break;
/*
#ifdef FULL_MAIN_BUFFER_SUPPORTED
		case JBUF_SAVE_SOURCE:
		case JBUF_CRANK_DEST:
		case JBUF_SAVE_AND_PASS:
			if (main->whole_image[0] == NULL)
				ERREXIT(cinfo, JERR_BAD_BUFFER_MODE);
			main->pub.process_data = process_data_buffer_main;
			break;
#endif
 */
	    default:
		throw new IOException(JPEGError.JERR_BAD_BUFFER_MODE);
	}
    }
    
    //=============================================================================
    // processData
    //=============================================================================
    /**
     * Replaces process_data function pointer in original C version.
     */
    //=============================================================================
    public int processData(JPEGImageSaver cinfo, int[] inputBuf,
    int inRowCtr, int inRowsAvail, int off, int scansize) throws IOException {
	switch ( processDataMethod ) {
	    case PROCESS_DATA_SIMPLE_MAIN:
		return processDataSimpleMain(cinfo, inputBuf, inRowCtr,
		inRowsAvail, off, scansize);
	    default:
		return 0;
	}
    }
    
    //=============================================================================
    // processDataSimpleMain
    //=============================================================================
    /**
     * Process some data.
     * This routine handles the simple pass-through mode,
     * where we have only a strip buffer.
     *
     * Changed from void to int because we return the new value of inRowCtr.
     *
     * See process_data_simple_main in jcmainct.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public int processDataSimpleMain(JPEGImageSaver cinfo, int[] inputBuf,
    int inRowCtr, int inRowsAvail, int off, int scansize) throws IOException {
	int[]	param_buf = new int[2];
	
/*
METHODDEF(void)
process_data_simple_main (j_compress_ptr cinfo,
			  JSAMPARRAY input_buf, JDIMENSION *in_row_ctr,
			  JDIMENSION in_rows_avail)
 */
	while ( curiMCURow < cinfo.totaliMCURows ) {
	    // Read input data if we haven't filled the main buffer yet.
	    if ( rowgroupCtr < JPEGImageSaver.DCTSIZE ) {
		param_buf[0] = inRowCtr;
		param_buf[1] = rowgroupCtr;
		cinfo.prep.preProcessData(cinfo, inputBuf, param_buf, inRowsAvail,
		buffer, (int) JPEGImageSaver.DCTSIZE, off, scansize);
		inRowCtr = param_buf[0];
		rowgroupCtr = param_buf[1];
	    }
	    
	    // If we don't have a full iMCU row buffered, return to application for
	    // more data.  Note that preprocessor will always pad to fill the iMCU row
	    // at the bottom of the image.
	    if ( rowgroupCtr != JPEGImageSaver.DCTSIZE )
		return inRowCtr;
	    
	    // Send the completed row to the compressor.
/*			if ( ! (*cinfo->coef->compress_data) (cinfo, main->buffer)) {
				// If compressor did not consume the whole row, then we must need to
				// suspend processing and return to the application.  In this situation
				// we pretend we didn't yet consume the last input row; otherwise, if
				// it happened to be the last row of the image, the application would
				// think we were done.
				if (! main->suspended) {
					(*in_row_ctr)--;
					main->suspended = TRUE;
				}
				return;
			}
 */
	    // We did finish the row.  Undo our little suspension hack if a previous
	    // call suspended; then mark the main buffer empty.
	    if ( suspended ) {
		inRowCtr++;
		suspended = false;
	    }
	    rowgroupCtr = 0;
	    curiMCURow++;
	}
	return inRowCtr;
    }
    
    // #ifdef FULL_MAIN_BUFFER_SUPPORTED
    
/*
 * Process some data.
 * This routine handles all of the modes that use a full-size buffer.
 */
/*
METHODDEF(void)
process_data_buffer_main (j_compress_ptr cinfo,
			  JSAMPARRAY input_buf, JDIMENSION *in_row_ctr,
			  JDIMENSION in_rows_avail)
{
  my_main_ptr main = (my_main_ptr) cinfo->main;
  int ci;
  jpeg_component_info *compptr;
  boolean writing = (main->pass_mode != JBUF_CRANK_DEST);
 
  while (main->cur_iMCU_row < cinfo->total_iMCU_rows) {
    // Realign the virtual buffers if at the start of an iMCU row.
    if (main->rowgroup_ctr == 0) {
      for (ci = 0, compptr = cinfo->comp_info; ci < cinfo->num_components;
	   ci++, compptr++) {
	main->buffer[ci] = (*cinfo->mem->access_virt_sarray)
	  ((j_common_ptr) cinfo, main->whole_image[ci],
	   main->cur_iMCU_row * (compptr->v_samp_factor * DCTSIZE),
	   (JDIMENSION) (compptr->v_samp_factor * DCTSIZE), writing);
      }
      // In a read pass, pretend we just read some source data.
      if (! writing) {
 *in_row_ctr += cinfo->max_v_samp_factor * DCTSIZE;
	main->rowgroup_ctr = DCTSIZE;
      }
    }
 
    // If a write pass, read input data until the current iMCU row is full.
    // Note: preprocessor will pad if necessary to fill the last iMCU row.
    if (writing) {
      (*cinfo->prep->pre_process_data) (cinfo,
					input_buf, in_row_ctr, in_rows_avail,
					main->buffer, &main->rowgroup_ctr,
					(JDIMENSION) DCTSIZE);
      // Return to application if we need more data to fill the iMCU row.
      if (main->rowgroup_ctr < DCTSIZE)
	return;
    }
 
    // Emit data, unless this is a sink-only pass.
    if (main->pass_mode != JBUF_SAVE_SOURCE) {
      if (! (*cinfo->coef->compress_data) (cinfo, main->buffer)) {
	// If compressor did not consume the whole row, then we must need to
	// suspend processing and return to the application.  In this situation
	// we pretend we didn't yet consume the last input row; otherwise, if
	// it happened to be the last row of the image, the application would
	// think we were done.
	if (! main->suspended) {
	  (*in_row_ctr)--;
	  main->suspended = TRUE;
	}
	return;
      }
 
      // We did finish the row.  Undo our little suspension hack if a previous
      // call suspended; then mark the main buffer empty.
      if (main->suspended) {
	(*in_row_ctr)++;
	main->suspended = FALSE;
      }
    }
 
    // If get here, we are done with this iMCU row.  Mark buffer empty.
    main->rowgroup_ctr = 0;
    main->cur_iMCU_row++;
  }
}
 */
    // #endif // FULL_MAIN_BUFFER_SUPPORTED
    
    
} // End of class.

