/*****************************************************************************
 *
 *                              JPEGImageSaver.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.util.Vector;
import java.util.Hashtable;
import java.awt.Toolkit;
import java.io.Serializable;
import java.util.zip.*;

import fi.faidon.util.ByteArrayConversion;

/*
JPEG logic
----------
 
1. Sanity check on image size.
2.
 
 
 */


/**
 * Class for saving an image in the JPEG format.
 *
 * @author Kary FR&Auml;MLING
 */
public class JPEGImageSaver extends ImageSaverInterface
implements Serializable, ImageConsumer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final String	FORMAT_CODE = "JPEG";
    public static final String	FORMAT_COMPLETE_NAME = "JPEG/JFIF";
    public static final String	FORMAT_EXTENSION = "jpg";
    
    //--------------------------------------------------------------------------------------
    // Protected constants.
    //--------------------------------------------------------------------------------------
    
    protected static final int	DCTSIZE = 8;		// The basic DCT block is 8x8 samples.
    protected static final int	DCTSIZE2 = 64;		// DCTSIZE squared; # of elements in a block.
    protected static final int	MAX_SAMP_FACTOR = 4;	// JPEG limit on sampling factors.
    protected static final int	MAX_COMPONENTS = 10;	// Maximum number of components allowed by JPEG.
    protected static final int	BITS_IN_JSAMPLE = 8;	// Number of bits/JPEG sample. We always use 8.
    // It may be 8 or 12.
    protected static final int	JPEG_MAX_DIMENSION = 65500;
    protected static final int	MAX_COMPS_IN_SCAN = 4;	// JPEG limit on # of components in one scan.
    protected static final int	C_MAX_BLOCKS_IN_MCU = 10;	// compressor's limit on blocks per MCU.
    
    /**
     * Operating modes for buffer controllers
     */
    protected static final int	JBUF_PASS_THRU = 0;		// Plain stripwise operation.
    
    /**
     * Remaining modes require a full-image buffer to have been created
     */
    protected static final int	JBUF_SAVE_SOURCE = 1;	// Run source subobject only, save output.
    protected static final int	JBUF_CRANK_DEST = 2;	// Run dest subobject only, using saved data.
    protected static final int	JBUF_SAVE_AND_PASS = 3;	// Run both subobjects, save output.
    
    //--------------------------------------------------------------------------------------
    // Protected fields.
    //--------------------------------------------------------------------------------------
    protected int		width;
    protected int		height;
    protected int		inputComponents;		// # of color components in input image.
    protected int		inColorSpace;			// colorspace of input image.
    protected int		numComponents;			// # of color components in JPEG image.
    protected int		dataPrecision;			// Bits of precision in image data.
    protected Vector	compInfoVector;			// compInfoVector[i] describes component that appears i'th in SOF.
    protected Vector	curCompInfo;			// curCompInfo[i] describes current component.
    protected int		jpegColorSpace;			// colorspace of JPEG image.
    protected boolean	optimizeCoding;			// Optimize coding or not (for dataPrecision > 8);
    protected boolean	rawDataIn;
    protected boolean	arithCode;				// Use arithmetic coding or not.
    protected boolean	CCIR601sampling;
    protected boolean	progressiveMode;		//
    protected int		numScans;
    protected Object	scanInfo;				// Should not be used in current implementation.
    protected int		compsInScan;			// This currently seems to always be equal to numComponents, to verify!
    protected int		maxHsampFactor;
    protected int		maxVsampFactor;
    protected int		totaliMCURows;
    protected int		MCUsPerRow;
    protected int		MCURowsInScan;
    protected int		blocksInMCU;
    protected int[]		MCUmembership = new int[C_MAX_BLOCKS_IN_MCU];
    protected int		Ss;
    protected int		Se;
    protected int		Ah;
    protected int		Al;
    protected int		restartInterval;
    protected int		restartInRows;
    protected JPEGColor	cconvert;				// Color converter object.
    protected JPEGSample	downsample;			// Downsampler object.
    protected JPEGMaster	master;				// Master object.
    protected JPEGPrepController	prep;		// Preprocessing controller object.
    protected JPEGMainController	main;		// Main controller object.
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    
    private final int	NUM_QUANT_TBLS = 4;		// Quantization tables are numbered 0..3
    private final int	NUM_HUFF_TBLS = 4;		// Huffman tables are numbered 0..3
    private final int	NUM_ARITH_TBLS = 16;	// Arith-coding tables are numbered 0..15
    
    // DCT/IDCT algorithm options.
    private final int JDCT_ISLOW = 0;			// slow but accurate integer algorithm.
    private final int JDCT_IFAST = 1;			// faster, less accurate integer method.
    private final int JDCT_FLOAT = 2;			// floating-point: accurate, fast on fast HW.
    private final int JDCT_DEFAULT = JDCT_ISLOW;	// Default algorithm used.
    
    // Dithering options for decompression.
    private final int JDITHER_NONE = 0;			// no dithering.
    private final int JDITHER_ORDERED = 1;		// simple ordered dither.
    private final int JDITHER_FS = 2;			// Floyd-Steinberg error diffusion dither.
    
    private final byte[]	JFIF_ID_STRING = {0x4A, 0x46, 0x49, 0x46, 0};
    
    // We currently emit version code 1.01 since we use no 1.02 features.
    // This may avoid complaints from some older decoders.
    private final byte[]	JFIF_VERSION = {0x01, 0x01};
    private final int	JFIF_APP0_MARKER_LENGTH = 16;
    private final byte	MARKER_START = (byte) 0xFF;
    
    // Start Of Frame N. N indicates which compression process.
    // Only SOF0-SOF2 are now in common use. NB: codes C4 and CC are NOT SOF markers.
    private final byte	M_SOF0 = (byte) 0xC0;				// Baseline
    private final byte	M_SOF1 = (byte) 0xC1;				// Extended sequential, Huffman
    private final byte	M_SOF2 = (byte) 0xC2;				// Progressive, Huffman
    private final byte	M_SOF3 = (byte) 0xC3;				// Lossless, Huffman
    private final byte	M_SOF5 = (byte) 0xC5;				// Differential sequential, Huffman
    private final byte	M_SOF6 = (byte) 0xC6;				// Differential progressive, Huffman
    private final byte	M_SOF7 = (byte) 0xC7;				// Differential lossless, Huffman
    private final byte	M_SOF9 = (byte) 0xC9;				// Extended sequential, arithmetic
    private final byte	M_SOF10 = (byte) 0xCA;				// Progressive, arithmetic
    private final byte	M_SOF11 = (byte) 0xCB;				// Lossless, arithmetic
    private final byte	M_SOF13 = (byte) 0xCD;				// Differential sequential, arithmetic
    private final byte	M_SOF14 = (byte) 0xCE;				// Differential progressive, arithmetic
    private final byte	M_SOF15 = (byte) 0xCF;				// Differential lossless, arithmetic
    
    private final byte	M_DHT = (byte) 0xC4;		// Huffman table.
    private final byte	M_DAC = (byte) 0xCC;		// Arithmetic conditioning.
    
    private final byte	M_RST0 = (byte) 0xD0;
    private final byte	M_RST1 = (byte) 0xD1;
    private final byte	M_RST2 = (byte) 0xD2;
    private final byte	M_RST3 = (byte) 0xD3;
    private final byte	M_RST4 = (byte) 0xD4;
    private final byte	M_RST5 = (byte) 0xD5;
    private final byte	M_RST6 = (byte) 0xD6;
    private final byte	M_RST7 = (byte) 0xD7;
    
    private final byte	M_SOI = (byte) 0xD8;		// Start Of Image (beginning of datastream)
    private final byte	M_EOI = (byte) 0xD9;		// End Of Image (end of datastream)
    private final byte	M_SOS = (byte) 0xDA;		// Start Of Scan (begins compressed data)
    private final byte	M_DQT = (byte) 0xDB;
    private final byte	M_DNL = (byte) 0xDC;
    private final byte	M_DRI = (byte) 0xDD;
    private final byte	M_DHP = (byte) 0xDE;
    private final byte	M_EXP = (byte) 0xDF;
    
    private final byte	M_APP0 = (byte) 0xE0;
    private final byte	M_APP1 = (byte) 0xE1;
    private final byte	M_APP2 = (byte) 0xE2;
    private final byte	M_APP3 = (byte) 0xE3;
    private final byte	M_APP4 = (byte) 0xE4;
    private final byte	M_APP5 = (byte) 0xE5;
    private final byte	M_APP6 = (byte) 0xE6;
    private final byte	M_APP7 = (byte) 0xE7;
    private final byte	M_APP8 = (byte) 0xE8;
    private final byte	M_APP9 = (byte) 0xE9;
    private final byte	M_APP10 = (byte) 0xEA;
    private final byte	M_APP11 = (byte) 0xEB;
    private final byte	M_APP12 = (byte) 0xEC;
    private final byte	M_APP13 = (byte) 0xED;
    private final byte	M_APP14 = (byte) 0xEE;
    private final byte	M_APP15 = (byte) 0xEF;
    
    private final byte	M_JPG0 = (byte) 0xF0;
    private final byte	M_JPG13 = (byte) 0xFD;
    private final byte	M_COM = (byte) 0xFE;		// COMment
    
    private final byte	M_TEM = (byte) 0x01;
    private final int	M_ERROR = 0x100;
    
    // State values.
    private final int	CSTATE_START = 100;		/* after create_compress */
    private final int	CSTATE_SCANNING = 101;	/* start_compress done, write_scanlines OK */
    private final int	CSTATE_RAW_OK = 102;	/* start_compress done, write_raw_data OK */
    private final int	CSTATE_WRCOEFS = 103;	/* jpeg_write_coefficients done */
    private final int	DSTATE_START = 200;		/* after create_decompress */
    private final int	DSTATE_INHEADER = 201;	/* reading header markers, no SOS yet */
    private final int	DSTATE_READY = 202;		/* found SOS, ready for start_decompress */
    private final int	DSTATE_PRELOAD = 203;	/* reading multiscan file in start_decompress*/
    private final int	DSTATE_PRESCAN = 204;	/* performing dummy pass for 2-pass quant */
    private final int	DSTATE_SCANNING = 205;	/* start_decompress done, read_scanlines OK */
    private final int	DSTATE_RAW_OK = 206;	/* start_decompress done, read_raw_data OK */
    private final int	DSTATE_BUFIMAGE = 207;	/* expecting jpeg_start_output */
    private final int	DSTATE_BUFPOST = 208;	/* looking for SOS/EOI in jpeg_finish_output */
    private final int	DSTATE_RDCOEFS = 209;	/* reading file in jpeg_read_coefficients */
    private final int	DSTATE_STOPPING = 210;	/* looking for EOI in jpeg_finish_decompress */
    
    // These are the sample quantization tables given in JPEG spec section K.1.
    // The spec says that the values given produce "good" quality, and
    // when divided by 2, "very good" quality.
    private final int[]	STD_LUMINANCE_QUANT_TBL = {
	16,  11,  10,  16,  24,  40,  51,  61,
	12,  12,  14,  19,  26,  58,  60,  55,
	14,  13,  16,  24,  40,  57,  69,  56,
	14,  17,  22,  29,  51,  87,  80,  62,
	18,  22,  37,  56,  68, 109, 103,  77,
	24,  35,  55,  64,  81, 104, 113,  92,
	49,  64,  78,  87, 103, 121, 120, 101,
	72,  92,  95,  98, 112, 100, 103,  99
    };
    
    private final int[]	 STD_CHROMINANCE_QUANT_TBL = {
	17,  18,  24,  47,  99,  99,  99,  99,
	18,  21,  26,  66,  99,  99,  99,  99,
	24,  26,  56,  99,  99,  99,  99,  99,
	47,  66,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99
    };
    
    // jpeg_natural_order[i] is the natural-order position of the i'th element
    // of zigzag order.
    //
    // When reading corrupted data, the Huffman decoders could attempt
    // to reference an entry beyond the end of this array (if the decoded
    // zero run length reaches past the end of the block).  To prevent
    // wild stores without adding an inner-loop test, we put some extra
    // "63"s after the real entries.  This will cause the extra coefficient
    // to be stored in location 63 of the block, not somewhere random.
    // The worst case would be a run-length of 15, which means we need 16
    // fake entries.
    private final int[]	 JPEG_NATURAL_ORDER = {
	0,  1,  8, 16,  9,  2,  3, 10,
	17, 24, 32, 25, 18, 11,  4,  5,
	12, 19, 26, 33, 40, 48, 41, 34,
	27, 20, 13,  6,  7, 14, 21, 28,
	35, 42, 49, 56, 57, 50, 43, 36,
	29, 22, 15, 23, 30, 37, 44, 51,
	58, 59, 52, 45, 38, 31, 39, 46,
	53, 60, 61, 54, 47, 55, 62, 63,
	63, 63, 63, 63, 63, 63, 63, 63, /* extra entries for safety in decoder */
	63, 63, 63, 63, 63, 63, 63, 63
    };
    
	/*
	 * Huffman coding tables.
	 */
    private final int[] BITS_DC_LUMINANCE =
    { /* 0-base */ 0, 0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0 };
    private final int[]	VAL_DC_LUMINANCE =
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 };
    
    private final int[]	BITS_DC_CHROMINANCE =
    { /* 0-base */ 0, 0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0 };
    private final int[]	VAL_DC_CHROMINANCE =
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 };
    
    private final int[]	BITS_AC_LUMINANCE =
    { /* 0-base */ 0, 0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 0x7d };
    private final int[]	VAL_AC_LUMINANCE =
    { 0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
      0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
      0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
      0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
      0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16,
      0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
      0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
      0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
      0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
      0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
      0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
      0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
      0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
      0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
      0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
      0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4,
      0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
      0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
      0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
      0xf9, 0xfa };
      
      private final int[]	BITS_AC_CHROMINANCE =
      { /* 0-base */ 0, 0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 0x77 };
      private final int[]	VAL_AC_CHROMINANCE =
      { 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
	0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
	0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
	0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0,
	0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34,
	0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
	0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38,
	0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
	0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
	0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96,
	0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5,
	0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
	0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
	0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2,
	0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
	0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
	0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xf9, 0xfa };
	
	//--------------------------------------------------------------------------------------
	// Private fields.
	//--------------------------------------------------------------------------------------
	private FileOutputStream	writeFileHandle;
	private int		saveStatus;
	private int		byteCount;
	private double	inputGamma;				// image gamma of input image.
	private int		quality;				// Compression quality value.
	private Vector	quantizationTables;		// Vector of quantization tables.
	private boolean[]	qtSentTable;		// Array of indicators for written quantization tables.
	// Stupid of me (KF), but there's no separate object for
	// quantization tables, so we do the testing with this
	// for multiple writing with this instead.
	private Vector	dcHuffTbls;				// ptrs to Huffman coding tables, or NULL if not defined.
	private Vector	acHuffTbls;
	//	private	cinfo->scan_info = NULL;
	private int		smoothingFactor;
	private int		dctMethod;
	private int		densityUnit;
	private int		Xdensity;
	private int		Ydensity;
	//  UINT8 arith_dc_L[NUM_ARITH_TBLS]; /* L values for DC arith-coding tables */
	//  UINT8 arith_dc_U[NUM_ARITH_TBLS]; /* U values for DC arith-coding tables */
	//  UINT8 arith_ac_K[NUM_ARITH_TBLS]; /* Kx values for AC arith-coding tables */
	private boolean	writeJFIFheader;		// Write JFIF format or not.
	private boolean	writeAdobeMarker;		// Write Adobe format or not.
	private int		nextScanline;			// Treated scanline counter.
	private int		globalState;			// State indicator.
	private int		bytesPerPixel;
	private boolean	headerIsWritten;
	private boolean	imageCompleteDone;
	
	//--------------------------------------------------------------------------------------
	// Public methods.
	//--------------------------------------------------------------------------------------
	
	//=============================================================================
	// Constructor
	//=============================================================================
	/**
	 * @author Kary FR&Auml;MLING 6/6/1998.
	 */
	//=============================================================================
	public JPEGImageSaver() {
	    // Nothing written yet.
	    headerIsWritten = false;
	    imageCompleteDone = false;
	}
	
	//=============================================================================
	/**
	 * ImageSaverInterface method implementations.
	 */
	//=============================================================================
	public String getFormatCode() { return FORMAT_CODE; }
	public String getFormatString() { return FORMAT_COMPLETE_NAME; }
	public String getFormatExtension() { return FORMAT_EXTENSION; }
	
	//=============================================================================
	// saveIt
	//=============================================================================
	/**
	 * Save the image.
	 *
	 * This follows more or less the write_JPEG_file function
	 * in Example.c of the IJG Jpeg-6a library.
	 */
	//=============================================================================
	public boolean saveIt() {
	    ImageProducer	ip;
	    
	    // Verify that we have an image.
	    if ( saveImage == null ) return false;
	    
	    // No status information yet.
	    saveStatus = 0;
	    
	    /* Step 1: allocate and initialize JPEG compression object */
	    // No need to do it, this object already is the object.
	    
	    /* Step 2: specify data destination (eg, a file) */
	    /* Note: steps 2 and 3 can be done in either order. */
	    // Open the file.
	    try {
		writeFileHandle = new FileOutputStream(savePath);
	    } catch ( IOException e ) { System.out.println("IOException occurred opening FileOutputStream : " + e); }
	    
	    // Return straight away if we couldn't get a handle.
	    if ( writeFileHandle == null ) return false;
	    
		/* Steps 3 to 7 are done at the first setPixals call, when we are sure
		 * to have the size of the image and all other needed parameters.
		 */
	    
	    // Set us up as image consumer and start producing.
	    ip = saveImage.getSource();
	    if ( ip == null ) return false;
	    ip.startProduction(this);
	    
	    // Nothing more to do, just get data and close file at the end.
	    return true;
	}
	
	//=============================================================================
	// checkSave
	//=============================================================================
	/**
	 * Return ImageObserver constants for indicating the state of the image saving.
	 *
	 * @author Kary FR&Auml;MLING
	 */
	//=============================================================================
	public int checkSave() {
	    return saveStatus;
	}
	
	//=============================================================================
	/**
	 * ImageConsumer method implementations.
	 */
	//=============================================================================
	public void setProperties(Hashtable props) {
	    saveStatus |= ImageObserver.PROPERTIES;
	}
	
	public void setHints(int hintflags) {}
	
	//=============================================================================
	// setColorModel
	//=============================================================================
	/**
	 * If the default color model is an indexed color model and if it has a
	 * transparency pixel, then we put in an alpha channel.
	 */
	//=============================================================================
	public void setColorModel(ColorModel model) {
	}
	
	//=============================================================================
	// setDimensions
	//=============================================================================
	/**
	 */
	//=============================================================================
	public void setDimensions(int w, int h) {
	    // Store width and height.
	    width = w;
	    height = h;
	    
	    // Update save status.
	    saveStatus |= ImageObserver.WIDTH | ImageObserver.HEIGHT;
	}
	
	//=============================================================================
	// setPixels
	//=============================================================================
	/**
	 * Write the pixels into the file as RGB data. For this to work correctly,
	 * pixels should be delivered in topdownleftright order with complete
	 * scanlines. If we have several lines, the lines should be complete scanlines,
	 * otherwise the saving fails.
	 * @see ImageConsumer.
	 */
	//=============================================================================
	public void setPixels(int x, int y, int w, int h,
	ColorModel model, byte pixels[], int off,
	int scansize) {
	    int		i, rgb;
	    int		transparent_pixel = -1;
	    int[]	int_pix_buf;
/*
		// Change alpha channel mode only if the header isn't written yet.
		// We only use an alpha channel if the model has a transparent color.
		// This is allright if only one color model is used or if the first
		// one used has a transparent color. Otherwise there will be no
		// alpha channel.
		if ( !headerIsWritten && model instanceof IndexColorModel ) {
			if ( ((IndexColorModel) model).getTransparentPixel() != -1 ) {
				includeAlphaChannel = true;
				bytesPerPixel = 4;
			}
		}
 
		// If we are using an alpha channel, then get the transparent pixel
		// index if any.
		if ( includeAlphaChannel && model instanceof IndexColorModel ) {
			transparent_pixel = ((IndexColorModel) model).getTransparentPixel();
		}
 */
	    // Transform the byte array into the corresponding RGB array.
	    int_pix_buf = new int[pixels.length];
	    for ( i = off ; i < pixels.length ; i++ ) {
		// Funny, we have to mask out the last byte for this to work correctly.
		int_pix_buf[i] = model.getRGB(pixels[i]&0xFF);
		
		// Set alpha channel value if applicable.
/*			if ( includeAlphaChannel && (pixels[i]&0xFF) == transparent_pixel )
				int_pix_buf[i] &= 0xFFFFFF; */
	    }
	    
	    // Call the int array method.
	    setPixels(x, y, w, h, ColorModel.getRGBdefault(), int_pix_buf, off, scansize);
	}
	
	//=============================================================================
	// setPixels
	//=============================================================================
	/**
	 * @see ImageConsumer.
	 */
	//=============================================================================
	public void setPixels(int x, int y, int w, int h,
	ColorModel model, int pixels[], int off,
	int scansize) {
	    int		i, j, buf_off, rgb, deflate_size, in_byte_off;
	    byte[]	buf;
	    byte[]	four_byte_buf = new byte[4];
	    int		r, g, b, alpha;
	    int		old_r, old_g, old_b;
	    
	    // Write out the header if it is not written yet.
	    if ( !headerIsWritten ) try { writeHeader(); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; return; }
	    
	    // Pass the pixel array to the compressor. TAKE OFFSET AND SCANSIZE INTO CONSIDERATION
	    // HERE IF NECESSARY!!!!
	    try {
		jpegWriteScanlines(pixels, 1, off, scansize);
	    } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
/*
		// Fill the pixel buffer to pack. We suppose that we always get
		// entire scanlines if "h" is bigger than 1. The "+ h" is because
		// we insert a filtering indication byte at the beginning of each
		// scanline.
		buf = new byte[w*h*bytesPerPixel + h];
		buf_off = 0;
		old_r = old_g = old_b = 0;
		for ( i = 0 ; i < h ; i++ ) {
			// Set the filtering byte in the beginning of the scanline.
			if ( x == 0 ) buf[buf_off++] = NO_FILTER;
//			if ( x == 0 ) buf[buf_off++] = SUB_FILTER;
 
			// Treat the scanline.
			for ( j = 0 ; j < w ; j++ ) {
				// Get the RGB value in one go.
				rgb = model.getRGB(pixels[off + i*scansize + j]);
 
				// Set red, green and blue components.
				r = ((rgb>>16)&0xFF);
				g = ((rgb>>8)&0xFF);
				b = (rgb&0xFF);
				buf[buf_off++] = (byte) (r&0xFF);
				buf[buf_off++] = (byte) (g&0xFF);
				buf[buf_off++] = (byte) (b&0xFF);
 
				// If we have an alpha channel to include, then add it after
				// the RGB values.
				if ( includeAlphaChannel ) {
					alpha = model.getAlpha(pixels[off + i*scansize + j]);
					buf[buf_off++] = (byte) (alpha&0xFF);
				}
			}
		}
 
		// Compress the data and write it out as it is compressed.
		// We do this in 512 byte chunks, because it seems like
		// doing one big setInput gives a stack overflow.
		for ( in_byte_off = 0 ; in_byte_off < buf.length ; in_byte_off += DEFLATE_INPUT_SIZE ) {
			pixmapDeflater.setInput(buf, in_byte_off, (int) Math.min(DEFLATE_INPUT_SIZE, buf.length - in_byte_off));
		    while ( !pixmapDeflater.needsInput() ) {
				deflate_size = pixmapDeflater.deflate(deflate_buf);
				try { writeFileHandle.write(deflate_buf, 0, deflate_size); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
				byteCount += deflate_size;
				idatDataSize += deflate_size;
				crcCounter.update(deflate_buf, 0, deflate_size);
		}
		}
 */
	    // Update save status.
	    saveStatus |= ImageObserver.SOMEBITS;
	}
	
	//=============================================================================
	// imageComplete
	//=============================================================================
	/**
	 * Get imageComplete message so that we can close the output file.
	 * @see ImageConsumer.
	 */
	//=============================================================================
	public void imageComplete(int status) {
	    byte[]	two_byte_buf = new byte[2];
	    
	    // Finish everything if it has not been done yet.
	    if ( !imageCompleteDone ) {
		// Write out the remaining pixel data and add the CRC of the IDAT chunk.
/*			pixmapDeflater.finish();
			while ( !pixmapDeflater.finished() ) {
				deflate_size = pixmapDeflater.deflate(deflate_buf);
				try { writeFileHandle.write(deflate_buf, 0, deflate_size); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
				byteCount += deflate_size;
				idatDataSize += deflate_size;
				crcCounter.update(deflate_buf, 0, deflate_size);
			}
			ByteArrayConversion.ulongAsBytesBE(crcCounter.getValue(), four_byte_buf, 0, four_byte_buf.length);
			try { writeFileHandle.write(four_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
			byteCount += four_byte_buf.length;
 */
/*			// Data length. 0 for IEND.
			ByteArrayConversion.uintAsBytesBE(0, four_byte_buf, 0, four_byte_buf.length);
			try { writeFileHandle.write(four_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
			byteCount += four_byte_buf.length;
 */
		// Write EOI marker.
		two_byte_buf[0] = MARKER_START;	two_byte_buf[1] = M_EOI;
		try { writeFileHandle.write(two_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount += two_byte_buf.length;
		
		// Close the file and reopen in random access mode. This is just for setting
		// the size of the IDAT chunk data.
		if ( writeFileHandle != null ) {
		    try { writeFileHandle.close(); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		}
/*			try {
				RandomAccessFile rf = new RandomAccessFile(savePath, "rw");
				rf.seek(idatChunkSizeOffset);
				ByteArrayConversion.ulongAsBytesBE(idatDataSize, four_byte_buf, 0, four_byte_buf.length);
				rf.write(four_byte_buf);
				rf.close();
			} catch ( IOException e ) { System.out.println("IOException occurred opening RandomAccessFile : " + e); }
 */
		// We are ready with imageComplete treatment.
		imageCompleteDone = true;
	    }
	    
	    // Update save status.
	    saveStatus |= ImageObserver.ALLBITS;
	}
	
	//=============================================================================
	// writeFrameHeader
	//=============================================================================
	/**
	 * Write frame header.
	 * This consists of DQT and SOFn markers.
	 * Note that we do not emit the SOF until we have emitted the DQT(s).
	 * This avoids compatibility problems with incorrect implementations that
	 * try to error-check the quant table numbers as soon as they see the SOF.
	 *
	 * see write_frame_header in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	public synchronized void writeFrameHeader() {
	    int		ci, prec;
	    boolean	is_baseline;
	    JPEGComponentInfo	comp;
	    
	    System.out.println("writeFrameHeader");
	    // Emit DQT for each quantization table.
	    // Note that emit_dqt() suppresses any duplicate tables.
	    prec = 0;
	    for ( ci = 0 ; ci < numComponents; ci++ ) {
		comp = (JPEGComponentInfo) compInfoVector.elementAt(ci);
		prec += emitDQT(comp.quantTblNo);
	    }
	    // now prec is nonzero iff there are any 16-bit quant tables.
	    
	    // Check for a non-baseline specification.
	    // Note we assume that Huffman table numbers won't be changed later.
	    if ( arithCode || progressiveMode || dataPrecision != 8 ) {
		is_baseline = false;
	    }
	    else {
		is_baseline = true;
		for ( ci = 0 ; ci < numComponents ; ci++ ) {
		    comp = (JPEGComponentInfo) compInfoVector.elementAt(ci);
		    if ( comp.dcTblNo > 1 || comp.acTblNo > 1 )
			is_baseline = false;
		}
		if ( prec > 0 && is_baseline ) {
		    is_baseline = false;
		    
		    // If it's baseline except for quantizer size, warn the user.
		    //				TRACEMS(cinfo, 0, JTRC_16BIT_TABLES);
		}
	    }
	    
	    // Emit the proper SOF marker.
	    if ( arithCode ) {
		emitSOF(M_SOF9);	// SOF code for arithmetic coding.
	    }
	    else {
		if ( progressiveMode )
		    emitSOF( M_SOF2 );	// SOF code for progressive Huffman.
		else if ( is_baseline )
		    emitSOF( M_SOF0 );	// SOF code for baseline implementation.
		else
		    emitSOF( M_SOF1 );	// SOF code for non-baseline Huffman file.
	    }
	}
	
	//=============================================================================
	// writeScanHeader
	//=============================================================================
	/**
	 * Write scan header.
	 * This consists of DHT or DAC markers, optional DRI, and SOS.
	 * Compressed data will be written following the SOS.
	 *
	 * see write_scan_header in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	public synchronized void writeScanHeader() {
	    int		i;
	    JPEGComponentInfo	comp;
	    
	    System.out.println("writeScanHeader");
	    // Set up the scan parameters. Added by KF.
	    //		selectScanParameters();
	    
	    if ( arithCode ) {
		// Emit arith conditioning info.  We may have some duplication
		// if the file has multiple scans, but it's so small it's hardly
		// worth worrying about.
		//			emit_dac(cinfo);
	    }
	    else {
		// Emit Huffman tables.
		// Note that emit_dht() suppresses any duplicate tables.
		System.out.println("compsInScan:" + compsInScan);
		for ( i = 0 ; i < compsInScan ; i++ ) {
		    comp = (JPEGComponentInfo) compInfoVector.elementAt(i);
		    if ( progressiveMode ) {
			// Progressive mode: only DC or only AC tables are used in one scan
/*					if (cinfo->Ss == 0) {
						if (cinfo->Ah == 0)		// DC needs no table for refinement scan.
							emit_dht(cinfo, compptr->dc_tbl_no, FALSE);
					} else {
						emit_dht(cinfo, compptr->ac_tbl_no, TRUE);
					} */
		    }
		    else {
			System.out.println("Sequential mode");
			// Sequential mode: need both DC and AC tables.
			emitDHT(comp.dcTblNo, false);
			emitDHT(comp.acTblNo, true);
		    }
		}
		
			/* Emit DRI if required --- note that DRI value could change for each scan.
			 * If it doesn't, a tiny amount of space is wasted in multiple-scan files.
			 * We assume DRI will never be nonzero for one scan and zero for a later one.
			 */
/*			if ( cinfo->restart_interval )
				emit_dri(cinfo);
 */
		emitSOS();
	    }
	}
	
	//=============================================================================
	// emit2Bytes
	//=============================================================================
	/**
	 * Emit a numeric value as two bytes.
	 */
	//=============================================================================
	public void emit2Bytes(int value) {
	    byte[]	byte_buf = new byte[2];
	    
	    ByteArrayConversion.uintAsBytesBE(value, byte_buf, 0, byte_buf.length);
	    try { writeFileHandle.write(byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf.length;
	}
	
	//=============================================================================
	// emitByte
	//=============================================================================
	/**
	 * Emit a numeric value as a byte.
	 */
	//=============================================================================
	public void emitByte(int value) {
	    byte[]	byte_buf = new byte[1];
	    
	    ByteArrayConversion.uintAsBytesBE(value, byte_buf, 0, byte_buf.length);
	    try { writeFileHandle.write(byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf.length;
	}
	
	
	//=============================================================================
	// jcopySampleRows
	//=============================================================================
	/**
	 * Copy some rows of samples from one place to another.
	 * num_rows rows are copied from input_array[source_row++]
	 * to output_array[dest_row++]; these areas may overlap for duplication.
	 * The source and destination arrays must be at least as wide as num_cols.
	 *
	 * This method is here just to avoid getting as many files as in the C
	 * implementation.
	 *
	 * see jcopy_sample_rows in JQUANT1.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	public static void jcopySampleRows(int[][] inputArray, int sourceRow,
	int[][] outputArray, int destRow,
	int numRows, int numCols) {
	    int		col, row;
	    
	    for ( row = numRows ; row > 0 ; row--) {
		for ( col = 0 ; col < numCols ; col++ )
		    outputArray[destRow][col] = inputArray[sourceRow][col];
		sourceRow++;
		destRow++;
	    }
	}
	
	//--------------------------------------------------------------------------------------
	// Private methods.
	//--------------------------------------------------------------------------------------
	
	//=============================================================================
	// writeHeader
	//=============================================================================
	/**
	 * Writes out the entire PNG header and the start of the IDAT chunk. It should
	 * be definitely known at this time if we use an alpha channel or not.
	 */
	//=============================================================================
	private synchronized void writeHeader() throws IOException {
	    int		i, off;
	    long	crc;
	    
	    // No bytes written yet.
	    byteCount = 0;
	    
	    System.out.println("writeHeader");
	    /* Step 3: set parameters for compression */
	    inputComponents = 3;		// # of color components per pixel.
	    inColorSpace = JPEGColor.JCS_RGB;		// Always RGB for Java AWT images.
	    
	    // Set up all default parameters.
	    jpegSetDefaults();
	    
	    // Now you can set any non-default parameters you wish to.
	    // Here we just illustrate the use of quality (quantization table) scaling:
	    //		jpegSetQuality(quality, true);	// limit to baseline-JPEG values.
	    
	    /* Step 4: Start compressor */
		/* TRUE ensures that we will write a complete interchange-JPEG file.
		 * Pass TRUE unless you are very sure of what you're doing.
		 */
	    jpegStartCompress(true);
	    
	    /* Step 5: while (scan lines remain to be written) */
	    /*           jpeg_write_scanlines(...); */
	    // This is done in the setPixels() methods.
	    
	    /* Step 6: Finish compression */
	    // This is done in the imageComplete() method.
	    
	    /* Step 7: release JPEG compression object */
	    // Automatically done by Java GC.
	    
	    // The header is written.
	    headerIsWritten = true;
	}
	
	//=============================================================================
	// writeFileHeader
	//=============================================================================
	/**
	 * Write datastream header.
	 * This consists of an SOI and optional APPn markers.
	 * We recommend use of the JFIF marker, but not the Adobe marker,
	 * when using YCbCr or grayscale data.  The JFIF marker should NOT
	 * be used for any other JPEG colorspace.  The Adobe marker is helpful
	 * to distinguish RGB, CMYK, and YCCK colorspaces.
	 * Note that an application can write additional header markers after
	 * jpeg_start_compress returns.
	 *
	 * see write_file_header in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private synchronized void writeFileHeader() {
	    // First the SOI.
	    emitMarker(M_SOI);
	    
	    // Next an optional JFIF APP0.
	    if ( writeJFIFheader )
		emitJfifApp0();
	    
	    // Next an optional Adobe APP14.
	    if ( writeAdobeMarker )
		emitAdobeApp14();
	}
	
	//=============================================================================
	// emitJfifApp0
	//=============================================================================
	/**
	 * Emit a JFIF-compliant APP0 marker.
	 *
	 * Length of APP0 block	(2 bytes)
	 * Block ID			(4 bytes - ASCII "JFIF")
	 * Zero byte			(1 byte to terminate the ID string)
	 * Version Major, Minor	(2 bytes - 0x01, 0x01)
	 * Units			(1 byte - 0x00 = none, 0x01 = inch, 0x02 = cm)
	 * Xdpu			(2 bytes - dots per unit horizontal)
	 * Ydpu			(2 bytes - dots per unit vertical)
	 * Thumbnail X size		(1 byte)
	 * Thumbnail Y size		(1 byte)
	 *
	 * see emit_jfif_app0 in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private synchronized void emitJfifApp0() {
	    // Write APP0 marker, data length and JFIF identifier string.
	    emitMarker(M_APP0);
	    
	    // APP0 data length.
	    emit2Bytes(JFIF_APP0_MARKER_LENGTH);
	    
	    // JFIF identifier string.
	    try { writeFileHandle.write(JFIF_ID_STRING); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += JFIF_ID_STRING.length;
	    
	    // JFIF version.
	    // We currently emit version code 1.01 since we use no 1.02 features.
	    // This may avoid complaints from some older decoders.
	    try { writeFileHandle.write(JFIF_VERSION); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += JFIF_VERSION.length;
	    
	    // Units. We could also emit 1 for pixels/inch and give screen resolution.
	    emitByte(densityUnit);
	    
	    // Xdensity and Ydensity. We use the current screen resolution as image
	    // resolution.
	    int screen_res = Toolkit.getDefaultToolkit().getScreenResolution();
	    emit2Bytes(Xdensity);
	    emit2Bytes(Ydensity);
	    
	    // No thumbnail.
	    emitByte(0);
	    emitByte(0);
	}
	
	//=============================================================================
	// emitAdobeApp14
	//=============================================================================
	/**
	 * Emit an Adobe APP14 marker.
	 *
	 * Length of APP14 block (2 bytes)
	 * Block ID (5 bytes - ASCII "Adobe")
	 * Version Number (2 bytes - currently 100)
	 * Flags0 (2 bytes - currently 0)
	 * Flags1 (2 bytes - currently 0)
	 * Color transform (1 byte)
	 *
	 * Although Adobe TN 5116 mentions Version = 101, all the Adobe files
	 * now in circulation seem to use Version = 100, so that's what we write.
	 *
	 * We write the color transform byte as 1 if the JPEG color space is
	 * YCbCr, 2 if it's YCCK, 0 otherwise.  Adobe's definition has to do with
	 * whether the encoder performed a transformation, which is pretty useless.
	 *
	 * see emit_adobe_app14 in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private synchronized void emitAdobeApp14() {
/*		emit_marker(cinfo, M_APP14);
 
		emit_2bytes(cinfo, 2 + 5 + 2 + 2 + 2 + 1); // length
 
		emit_byte(cinfo, 0x41);	// Identifier: ASCII "Adobe"
		emit_byte(cinfo, 0x64);
		emit_byte(cinfo, 0x6F);
		emit_byte(cinfo, 0x62);
		emit_byte(cinfo, 0x65);
		emit_2bytes(cinfo, 100);	// Version
		emit_2bytes(cinfo, 0);	// Flags0
		emit_2bytes(cinfo, 0);	// Flags1
		switch (cinfo->jpeg_color_space) {
		case JCS_YCbCr:
			emit_byte(cinfo, 1);	// Color transform = 1
			break;
		case JCS_YCCK:
			emit_byte(cinfo, 2);	// Color transform = 2
			break;
		default:
			emit_byte(cinfo, 0);	// Color transform = 0
			break;
		}
 */
	}
	
	//=============================================================================
	// emitDQT
	//=============================================================================
	/**
	 * Writes out a DQT marker and associated data.
	 * Returns the precision used (0 = 8bits, 1 = 16bits) for baseline checking.
	 */
	//=============================================================================
	private synchronized int emitDQT(int dqtTableIndex) {
	    int		i, data_length;
	    int		precision;
	    byte[]	one_byte_buf = new byte[1];
	    byte[]	two_byte_buf = new byte[2];
	    int[]	dqt_table;
	    
	    // Verify that we have quantization tables. We should actually send an
	    // exception if not.
	    if ( quantizationTables == null ) return 0;
	    dqt_table = (int[]) quantizationTables.elementAt(dqtTableIndex);
	    if ( dqt_table == null ) return 0;
	    
	    // Precision, values are 0 or 1 (1 byte). If we have any values > 255 in the
	    // quantization table, then the precision is 2 bytes (value 1). TO DO!
	    precision = 0;
	    for (i = 0 ; i < dqt_table.length ; i++) {
		if ( dqt_table[i] > 255) precision = 1;
	    }
	    
	    if ( !qtSentTable[dqtTableIndex] ) {
		
		// Write DQT marker.
		emitMarker(M_DQT);
		
		// Data length (2 bytes).
		data_length = 1 + 2;
		data_length += ( precision == 1 ) ? DCTSIZE2*2 : DCTSIZE2;
		emit2Bytes(data_length);
		
		// Write out DQT table index.
		emitByte(dqtTableIndex);
		
		// For each table entry 2 bytes if precision == 1, one byte otherwise:
		// - Quantification value (high byte).
		// - Quantification value (low byte).
		// Values are written in ZIGZAG ORDER!
		for ( i = 0 ; i < dqt_table.length ; i++ ) {
		    if ( precision == 0 ) {
			emitByte(dqt_table[JPEG_NATURAL_ORDER[i]]);
		    }
		    else {
			emit2Bytes(dqt_table[JPEG_NATURAL_ORDER[i]]);
		    }
		}
		
		// Write it only once.
		qtSentTable[dqtTableIndex] = true;
	    }
	    
	    return precision;
	}
	
	//=============================================================================
	// emitDHT
	//=============================================================================
	/**
	 * Writes out a DHT marker and associated data.
	 *
	 * See emit_dht in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private synchronized void emitDHT(int index, boolean isAc) {
	    int		i, length;
	    byte[]	one_byte_buf = new byte[1];
	    byte[]	two_byte_buf = new byte[2];
	    Vector	htables;
	    JpegHuffmanTbl	htbl;
	    
	    System.out.println("emitDHT");
	    // Get the Huffman table.
	    if ( isAc )
		htables = acHuffTbls;
	    else
		htables = dcHuffTbls;
	    if ( index < htables.size() ) {
		htbl = (JpegHuffmanTbl) htables.elementAt(index);
		if ( htbl == null ) { saveStatus = ImageObserver.ERROR; return; }
	    }
	    else {
		saveStatus = ImageObserver.ERROR;
		return;
	    }
	    if ( isAc ) index += 0x10;		// output index has AC bit set.
	    
	    // Return without output if the table has already been written.
	    if ( htbl.sentTable) return;
	    
	    // Write DHT marker.
	    two_byte_buf[0] = MARKER_START;	two_byte_buf[1] = M_DHT;
	    try { writeFileHandle.write(two_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += two_byte_buf.length;
	    
	    // Get Huffman table length.
	    length = 0;
	    for (i = 1; i <= 16; i++)
		length += htbl.bits[i];
	    
	    // Data length (2 bytes).
	    emit2Bytes(length + 2 + 1 + 16);
	    
	    // Huffman table index.
	    emitByte(index);
	    
	    // Huffman table bit bytes.
	    for (i = 1; i <= 16; i++) {
		one_byte_buf[0] = (byte) htbl.bits[i];
		try { writeFileHandle.write(one_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount ++;
	    }
	    
	    // Huffman table values.
	    for (i = 0; i < length; i++) {
		one_byte_buf[0] = (byte) htbl.huffval[i];
		try { writeFileHandle.write(one_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount ++;
	    }
	    // Mark the table as sent.
	    htbl.sentTable = true;
	}
	
	//=============================================================================
	// emitSOF
	//=============================================================================
	/**
	 * Writes out a SOFn marker and associated data.
	 */
	//=============================================================================
	private synchronized void emitSOF(byte sofCode) {
	    int		i, data_length;
	    int		precision;
	    byte	b;
	    byte[]	one_byte_buf = new byte[1];
	    
	    // Write SOFn marker.
	    emitMarker(sofCode);
	    
	    // Data length (2 bytes).
	    data_length = 3*numComponents + 2 + 5 + 1;
	    emit2Bytes(data_length);
	    
	    // Data precision (8 bits by default).
	    emitByte(dataPrecision);
	    
	    // Width & height.
	    emit2Bytes(height);
	    emit2Bytes(width);
	    
	    // Number of components (3 for RGB).
	    emitByte(numComponents);
	    
	    // After: 3 bytes per component, N components. TO DO!
	    // - Component ID.
	    // - Horizontal sampling factor (&0xF0), vertical sampling factor (&0xF).
	    // - Quantification table number.
	    JPEGComponentInfo comp;
	    for ( i = 0 ; i < numComponents ; i++ ) {
		// Get component.
		if ( compInfoVector != null ) {
		    comp = (JPEGComponentInfo) compInfoVector.elementAt(i);
		    if ( comp != null ) {
			// Component number, starts with 1.
			emitByte(comp.componentIndex);
			
			// Horizontal and vertical sampling factors coded into the same byte.
			one_byte_buf[0] = (byte) ((comp.hSampFactor << 4) + comp.vSampFactor);
			try { writeFileHandle.write(one_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
			byteCount++;
			
			// Quantization table number.
			emitByte(comp.quantTblNo);
		    }
		}
	    }
	}
	
	//=============================================================================
	// emitSOS
	//=============================================================================
	/**
	 * Emit a SOS marker.
	 *
	 * See emit_sos in jcmarker.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private synchronized void emitSOS() {
	    int		i, td, ta, data_length;
	    byte[]	one_byte_buf = new byte[1];
	    byte[]	two_byte_buf = new byte[2];
	    JPEGComponentInfo	comp;
	    
	    // Write SOFn marker.
	    two_byte_buf[0] = MARKER_START;	two_byte_buf[1] = M_SOS;
	    try { writeFileHandle.write(two_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += two_byte_buf.length;
	    
	    // Data length (2 bytes).
	    data_length = 2*compsInScan + 2 + 1 + 3;
	    ByteArrayConversion.uintAsBytesBE(data_length, two_byte_buf, 0, two_byte_buf.length);
	    try { writeFileHandle.write(two_byte_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += two_byte_buf.length;
	    
	    // Number of components in scan.
	    emitByte(compsInScan);
	    
	    // Write out component infos.
	    for ( i = 0 ; i < compsInScan ; i++ ) {
		comp = (JPEGComponentInfo) compInfoVector.elementAt(i);
		emitByte(comp.componentIndex);
		td = comp.dcTblNo;
		ta = comp.acTblNo;
		if ( progressiveMode ) {
		    // Progressive mode: only DC or only AC tables are used in one scan;
		    // furthermore, Huffman coding of DC refinement uses no table at all.
		    // We emit 0 for unused field(s); this is recommended by the P&M text
		    // but does not seem to be specified in the standard.
		    if ( Ss == 0 ) {
			ta = 0;			// DC scan.
			if ( Ah != 0 && !arithCode )
			    td = 0;		// no DC table either.
		    }
		    else {
			td = 0;			// AC scan.
		    }
		}
		emitByte((td << 4) + ta);
	    }
	    
	    emitByte(Ss);
	    emitByte(Se);
	    emitByte((Ah << 4) + Al);
	}
	
	//=============================================================================
	// emitMarker
	//=============================================================================
	/**
	 * Emit a marker code.
	 */
	//=============================================================================
	private void emitMarker(int mark) {
	    emitByte(0xFF);
	    emitByte(mark);
	}
	
	//=============================================================================
	// jpegSetDefaults
	//=============================================================================
	/**
	 * Default parameter setup for compression.
	 *
	 * See jpeg_set_defaults in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegSetDefaults() {
	    int		i;
	    
	    // Safety check to ensure start_compress not called yet.
/*		if (cinfo->global_state != CSTATE_START)
			ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
 */
	    // Set up a vector of component objects.
	    if ( compInfoVector == null ) {
		compInfoVector = new Vector(MAX_COMPONENTS);
		curCompInfo = new Vector(MAX_COMPONENTS);
	    }
	    for ( i = 0 ; i < MAX_COMPONENTS ; i++ ) {
		compInfoVector.addElement(null);
		curCompInfo.addElement(null);
	    }
	    
	    bytesPerPixel = 3;
	    
	    // Initialize everything not dependent on the color space.
	    dataPrecision = BITS_IN_JSAMPLE;
	    
	    // Set up two quantization tables using default quality of 75.
	    jpegSetQuality(75, true);
	    
	    // Set up two Huffman tables.
	    if ( dcHuffTbls == null ) dcHuffTbls = new Vector(NUM_HUFF_TBLS);
	    if ( acHuffTbls == null ) acHuffTbls = new Vector(NUM_HUFF_TBLS);
	    stdHuffTables();
	    
	    // Initialize default arithmetic coding conditioning.
/*		for (i = 0; i < NUM_ARITH_TBLS; i++) {
			cinfo->arith_dc_L[i] = 0;
			cinfo->arith_dc_U[i] = 1;
			cinfo->arith_ac_K[i] = 5;
		}
 */
	    // Default is no multiple-scan output.
	    /*		cinfo->scan_info = NULL;  */
	    numScans = 0;
	    
	    // Expect normal source image, not raw downsampled data.
	    rawDataIn = false;
	    
	    // Use Huffman coding, not arithmetic coding, by default.
	    arithCode = false;
	    
	    // By default, don't do extra passes to optimize entropy coding.
	    optimizeCoding = false;
	    
	    // The standard Huffman tables are only valid for 8-bit data precision.
	    // If the precision is higher, force optimization on so that usable
	    // tables will be computed.  This test can be removed if default tables
	    // are supplied that are valid for the desired precision.
	    if ( dataPrecision > 8 )
		optimizeCoding = true;
	    
	    // By default, use the simpler non-cosited sampling alignment.
	    CCIR601sampling = false;
	    
	    // No input smoothing.
	    smoothingFactor = 0;
	    
	    // DCT algorithm preference.
	    dctMethod = JDCT_DEFAULT;
	    
	    // No restart markers.
	    restartInterval = 0;
	    restartInRows = 0;
	    
	    // Fill in default JFIF marker parameters.  Note that whether the marker
	    // will actually be written is determined by jpeg_set_colorspace.
	    densityUnit = 0;	// Pixel size is unknown by default.
	    Xdensity = 1;		// Pixel aspect ratio is square by default.
	    Ydensity = 1;
	    
	    // Choose JPEG colorspace based on input space, set defaults accordingly.
	    jpegDefaultColorspace();
	}
	
	//=============================================================================
	// jpegSetQuality
	//=============================================================================
	/**
	 * Set or change the 'quality' (quantization) setting, using default tables.
	 * This is the standard quality-adjusting entry point for typical user
	 * interfaces; only those who want detailed control over quantization tables
	 * would use the preceding three routines directly.
	 *
	 * See jpeg_set_quality in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegSetQuality(int quality, boolean forceBaseline) {
	    // Convert user 0-100 rating to percentage scaling.
	    quality = jpegQualityScaling(quality);
	    
	    // Set up standard quality tables.
	    jpegSetLinearQuality(quality, forceBaseline);
	}
	
	//=============================================================================
	// jpegQualityScaling
	//=============================================================================
	/**
	 * Convert a user-specified quality rating to a percentage scaling factor
	 * for an underlying quantization table, using our recommended scaling curve.
	 * The input 'quality' factor should be 0 (terrible) to 100 (very good).
	 *
	 * See jpeg_quality_scaling in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private int jpegQualityScaling(int quality) {
	    // Safety limit on quality factor. Convert 0 to 1 to avoid zero divide.
	    if (quality <= 0) quality = 1;
	    if (quality > 100) quality = 100;
	    
	    // The basic table is used as-is (scaling 100) for a quality of 50.
	    // Qualities 50..100 are converted to scaling percentage 200 - 2*Q;
	    // note that at Q=100 the scaling is 0, which will cause jpeg_add_quant_table
	    // to make all the table entries 1 (hence, minimum quantization loss).
	    // Qualities 1..50 are converted to scaling percentage 5000/Q.
	    if (quality < 50)
		quality = 5000 / quality;
	    else
		quality = 200 - quality*2;
	    
	    return quality;
	}
	
	//=============================================================================
	// jpegSetLinearQuality
	//=============================================================================
	/**
	 * Set or change the 'quality' (quantization) setting, using default tables
	 * and a straight percentage-scaling quality scale.  In most cases it's better
	 * to use jpeg_set_quality (above); this entry point is provided for
	 * applications that insist on a linear percentage scaling.
	 *
	 * See jpeg_set_linear_quality in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegSetLinearQuality(int scaleFactor, boolean forceBaseline) {
	    // Set up two quantization tables using the specified scaling.
	    jpegAddQuantTable(0, STD_LUMINANCE_QUANT_TBL,
	    scaleFactor, forceBaseline);
	    jpegAddQuantTable(1, STD_CHROMINANCE_QUANT_TBL,
	    scaleFactor, forceBaseline);
	}
	
	//=============================================================================
	// jpegAddQuantTable
	//=============================================================================
	/**
	 * Define a quantization table equal to the basic_table times
	 * a scale factor (given as a percentage).
	 * If force_baseline is TRUE, the computed quantization table entries
	 * are limited to 1..255 for JPEG baseline compatibility.
	 *
	 * See jpeg_add_quant_table in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegAddQuantTable(int whichTbl, int[] basicTable, int scaleFactor,
	boolean forceBaseline) {
	    int		i;
	    long	temp;
	    int[]	quant_tbl;
	    
	    // Safety check to ensure start_compress not called yet.
/*		if (cinfo->global_state != CSTATE_START)
			ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
 */
	    // Get the existing quantization table if we have one or create a new
	    // one if not. We assume one byte/quantization value.
	    if ( quantizationTables == null ) {
		quantizationTables = new Vector(NUM_QUANT_TBLS);
		for ( i = 0 ; i < NUM_QUANT_TBLS ; i++ ) quantizationTables.addElement(null);
		
		// Initialize sent_table FALSE so table will be written to JPEG file.
		// KF: Done automatically by Java.
		qtSentTable = new boolean[NUM_QUANT_TBLS];
	    }
	    quant_tbl = new int[DCTSIZE2];
	    quantizationTables.setElementAt(quant_tbl, whichTbl);
	    
	    // Fill out the quantization table depending on the scale factor.
	    for (i = 0 ; i < quant_tbl.length ; i++) {
		temp = ((long) basicTable[i]*scaleFactor + 50L) / 100L;
		
		// limit the values to the valid range.
		if ( temp <= 0L ) temp = 1L;
		if ( temp > 32767L ) temp = 32767L;		// max quantizer needed for 12 bits.
		if ( forceBaseline && temp > 255L )
		    temp = 255L;						// limit to baseline range if requested.
		quant_tbl[i] = (int) (temp&0xFFFF);
	    }
	}
	
	//=============================================================================
	// jpegDefaultColorspace
	//=============================================================================
	/**
	 * Select an appropriate JPEG colorspace for inColorSpace.
	 *
	 * See jpeg_default_colorspace in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegDefaultColorspace() {
	    switch ( inColorSpace ) {
		case JPEGColor.JCS_GRAYSCALE:
		    jpegSetColorspace(JPEGColor.JCS_GRAYSCALE);
		    break;
		case JPEGColor.JCS_RGB:
		    jpegSetColorspace(JPEGColor.JCS_YCbCr);
		    break;
		case JPEGColor.JCS_YCbCr:
		    jpegSetColorspace(JPEGColor.JCS_YCbCr);
		    break;
		case JPEGColor.JCS_CMYK:
		    jpegSetColorspace(JPEGColor.JCS_CMYK); /* By default, no translation */
		    break;
		case JPEGColor.JCS_YCCK:
		    jpegSetColorspace(JPEGColor.JCS_YCCK);
		    break;
		case JPEGColor.JCS_UNKNOWN:
		    jpegSetColorspace(JPEGColor.JCS_UNKNOWN);
		    break;
		default:
		    //			ERREXIT(cinfo, JERR_BAD_IN_COLORSPACE);
	    }
	}
	
	//=============================================================================
	// jpegSetColorspace
	//=============================================================================
	/**
	 * Set the JPEG colorspace, and choose colorspace-dependent default values.
	 *
	 * See jpeg_set_colorspace in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegSetColorspace(int colorspace) {
	    int		ci;
	    JPEGComponentInfo	comp;
	    
	    // Safety check to ensure start_compress not called yet.
	    //		if (cinfo->global_state != CSTATE_START)
	    //			ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
	    
	    // For all colorspaces, we use Q and Huff tables 0 for luminance components,
	    // tables 1 for chrominance components.
	    jpegColorSpace = colorspace;
	    
	    writeJFIFheader = false;			// No marker for non-JFIF colorspaces.
	    writeAdobeMarker = false;			// write no Adobe marker by default.
	    
	    switch (colorspace) {
		case JPEGColor.JCS_GRAYSCALE:
		    writeJFIFheader = true;			// Write a JFIF marker.
		    numComponents = 1;
		    
		    // JFIF specifies component ID 1.
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 0);
		    comp.setComp(0, 1, 1,1, 0, 0,0);
		    break;
		case JPEGColor.JCS_RGB:
		    writeAdobeMarker = true;		// Write Adobe marker to flag RGB.
		    numComponents = 3;
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 0);
		    comp.setComp(0, 0x52 /* 'R' */, 1,1, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 1);
		    comp.setComp(1, 0x47 /* 'G' */, 1,1, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 2);
		    comp.setComp(2, 0x42 /* 'B' */, 1,1, 0, 0,0);
		    break;
		case JPEGColor.JCS_YCbCr:
		    writeJFIFheader = true;			// Write a JFIF marker.
		    numComponents = 3;
		    /* JFIF specifies component IDs 1,2,3 */
		    /* We default to 2x2 subsamples of chrominance */
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 0);
		    comp.setComp(0, 1, 2,2, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 1);
		    comp.setComp(1, 2, 1,1, 1, 1,1);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 2);
		    comp.setComp(2, 3, 1,1, 1, 1,1);
		    break;
		case JPEGColor.JCS_CMYK:
		    writeAdobeMarker = true;		// Write Adobe marker to flag CMYK.
		    numComponents = 4;
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 0);
		    comp.setComp(0, 0x43 /* 'C' */, 1,1, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 1);
		    comp.setComp(1, 0x4D /* 'M' */, 1,1, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 2);
		    comp.setComp(2, 0x59 /* 'Y' */, 1,1, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 3);
		    comp.setComp(3, 0x4B /* 'K' */, 1,1, 0, 0,0);
		    break;
		case JPEGColor.JCS_YCCK:
		    writeAdobeMarker = true;		// Write Adobe marker to flag YCCK.
		    numComponents = 4;
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 0);
		    comp.setComp(0, 1, 2,2, 0, 0,0);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 1);
		    comp.setComp(1, 2, 1,1, 1, 1,1);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 2);
		    comp.setComp(2, 3, 1,1, 1, 1,1);
		    comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, 3);
		    comp.setComp(3, 4, 2,2, 0, 0,0);
		    break;
		case JPEGColor.JCS_UNKNOWN:
		    numComponents = inputComponents;
		    if ( numComponents < 1 || numComponents > MAX_COMPONENTS) {
			//				ERREXIT2(cinfo, JERR_COMPONENT_COUNT, cinfo->num_components,
			//					MAX_COMPONENTS);
		    }
		    for ( ci = 0 ; ci < numComponents ; ci++ ) {
			comp = new JPEGComponentInfo(); compInfoVector.setElementAt(comp, ci);
			comp.setComp(ci, ci, 1,1, 0, 0,0);
		    }
		    break;
		default:
		    //			ERREXIT(cinfo, JERR_BAD_J_COLORSPACE);
	    }
	}
	
	//=============================================================================
	// stdHuffTables
	//=============================================================================
	/**
	 * Set up the standard Huffman tables (cf. JPEG standard section K.3).
	 * IMPORTANT: these are only valid for 8-bit data precision!
	 *
	 * See std_huff_tables in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void stdHuffTables() {
	    addHuffTable(dcHuffTbls, 0,
	    BITS_DC_LUMINANCE, VAL_DC_LUMINANCE);
	    addHuffTable(acHuffTbls, 0,
	    BITS_AC_LUMINANCE, VAL_AC_LUMINANCE);
	    addHuffTable(dcHuffTbls, 1,
	    BITS_DC_CHROMINANCE, VAL_DC_CHROMINANCE);
	    addHuffTable(acHuffTbls, 1,
	    BITS_AC_CHROMINANCE, VAL_AC_CHROMINANCE);
	}
	
	//=============================================================================
	// addHuffTable
	//=============================================================================
	/**
	 * Define a Huffman table.
	 *
	 * See add_huff_table in jcparam.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void addHuffTable(Vector tblVect, int tblIndex, int[] bits, int[] val) {
	    int		i;
	    JpegHuffmanTbl	tbl;
	    
	    // Set or insert the table into the vector.
	    tbl = new JpegHuffmanTbl();
	    if ( tblIndex < tblVect.size() )
		tblVect.setElementAt(tbl, tblIndex);
	    else
		tblVect.insertElementAt(tbl, tblIndex);
	    
	    // Make a copy of bits and val.
	    for ( i = 0 ; i < tbl.bits.length && i < bits.length ; i++ )
		tbl.bits[i] = bits[i];
	    for ( i = 0 ; i < tbl.huffval.length && i < val.length ; i++ )
		tbl.huffval[i] = val[i];
	    
	    // Initialize sent_table FALSE so table will be written to JPEG file.
	    //		(*htblptr)->sent_table = FALSE;
	}
	
	//*****************************************************************************
	// JCAPIMIN.C methods
	//*****************************************************************************
	
	//=============================================================================
	// jpegSuppressTables
	//=============================================================================
	/**
	 * Forcibly suppress or un-suppress all quantization and Huffman tables.
	 * Marks all currently defined tables as already written (if suppress)
	 * or not written (if !suppress).  This will control whether they get emitted
	 * by a subsequent jpeg_start_compress call.
	 *
	 * This routine is exported for use by applications that want to produce
	 * abbreviated JPEG datastreams.  It logically belongs in jcparam.c, but
	 * since it is called by jpeg_start_compress, we put it here --- otherwise
	 * jcparam.o would be linked whether the application used it or not.
	 *
	 * See jpeg_suppress_tables in jcapimin.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegSuppressTables(boolean suppress) {
	    int		i;
	    //		JQUANT_TBL * qtbl;
	    JpegHuffmanTbl	htbl;
	    
/*		for (i = 0; i < NUM_QUANT_TBLS; i++) {
			if ((qtbl = cinfo->quant_tbl_ptrs[i]) != NULL)
				qtbl->sent_table = suppress;
		}
 */
	    if ( dcHuffTbls != null ) {
		for ( i = 0; i < dcHuffTbls.size() ; i++) {
		    htbl = (JpegHuffmanTbl) dcHuffTbls.elementAt(i);
		    if ( htbl != null)
			htbl.sentTable = suppress;
		}
	    }
	    
	    if ( acHuffTbls != null ) {
		for ( i = 0; i < acHuffTbls.size() ; i++) {
		    htbl = (JpegHuffmanTbl) acHuffTbls.elementAt(i);
		    if ( htbl != null)
			htbl.sentTable = suppress;
		}
	    }
	}
	
	
	//*****************************************************************************
	// JCAPISTD.C methods
	//*****************************************************************************
	
	//=============================================================================
	// jpegStartCompress
	//=============================================================================
	/**
	 * Compression initialization.
	 * Before calling this, all parameters and a data destination must be set up.
	 *
	 * We require a write_all_tables parameter as a failsafe check when writing
	 * multiple datastreams from the same compression object.  Since prior runs
	 * will have left all the tables marked sent_table=TRUE, a subsequent run
	 * would emit an abbreviated stream (no tables) by default.  This may be what
	 * is wanted, but for safety's sake it should not be the default behavior:
	 * programmers should have to make a deliberate choice to emit abbreviated
	 * images.  Therefore the documentation and examples should encourage people
	 * to pass write_all_tables=TRUE; then it will take active thought to do the
	 * wrong thing.
	 *
	 * See jpeg_start_compress in jcapistd.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jpegStartCompress(boolean writeAllTables) throws IOException {
	    //		if (cinfo->global_state != CSTATE_START)
	    //			ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
	    
	    System.out.println("jpegStartCompress");
	    if (writeAllTables)
		jpegSuppressTables(false);		// Mark all tables to be written.
	    
	    // (Re)initialize error mgr and destination modules.
	    //		(*cinfo->err->reset_error_mgr) ((j_common_ptr) cinfo);
	    //		(*cinfo->dest->init_destination) (cinfo);
	    
	    // Perform master selection of active modules.
	    jinitCompressMaster();
	    
	    // Set up for the first pass.
	    master.prepareForPass(this);
	    
	    // Ready for application to drive first pass through jpeg_write_scanlines
	    // or jpeg_write_raw_data.
	    nextScanline = 0;
	    globalState = (rawDataIn ? CSTATE_RAW_OK : CSTATE_SCANNING);
	}
	
	//=============================================================================
	// jpegWriteScanlines
	//=============================================================================
	/**
	 * Write some scanlines of data to the JPEG compressor.
	 *
	 * The return value will be the number of lines actually written.
	 * This should be less than the supplied num_lines only in case that
	 * the data destination module has requested suspension of the compressor,
	 * or if more than image_height scanlines are passed in.
	 *
	 * Note: we warn about excess calls to jpeg_write_scanlines() since
	 * this likely signals an application programmer error.  However,
	 * excess scanlines passed in the last valid call are *silently* ignored,
	 * so that the application need not adjust num_lines for end-of-image
	 * when using a multiple-scanline buffer.
	 *
	 * See jpeg_write_scanlines in jcapistd.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private int jpegWriteScanlines(int[] scanlines, int numLines, int off,
	int scansize) throws IOException {
	    int		row_ctr, rows_left;
	    
/*		if (cinfo->global_state != CSTATE_SCANNING)
			ERREXIT1(cinfo, JERR_BAD_STATE, cinfo->global_state);
		if (cinfo->next_scanline >= cinfo->image_height)
			WARNMS(cinfo, JWRN_TOO_MUCH_DATA);
 */
	    // Call progress monitor hook if present.
/*		if (cinfo->progress != NULL) {
			cinfo->progress->pass_counter = (long) cinfo->next_scanline;
			cinfo->progress->pass_limit = (long) cinfo->image_height;
			(*cinfo->progress->progress_monitor) ((j_common_ptr) cinfo);
		}
 */
	    // Give master control module another chance if this is first call to
	    // jpeg_write_scanlines.  This lets output of the frame/scan headers be
	    // delayed so that application can write COM, etc, markers between
	    // jpeg_start_compress and jpeg_write_scanlines.
	    if ( master.callPassStartup)
		master.passStartup(this);
	    
	    // Ignore any extra scanlines at bottom of image.
	    rows_left = height - nextScanline;
	    if ( numLines > rows_left)
		numLines = rows_left;
	    
	    row_ctr = 0;
	    row_ctr = main.processData(this, scanlines, row_ctr, numLines, off, scansize);
	    nextScanline += row_ctr;
	    return row_ctr;
	}
	
	//*****************************************************************************
	// JCINIT.C methods
	//*****************************************************************************
	
	//=============================================================================
	// jinitCompressMaster
	//=============================================================================
	/**
	 * Master selection of compression modules.
	 * This is done once at the start of processing an image.  We determine
	 * which modules will be used and give them appropriate initialization calls.
	 *
	 * See jinit_compress_master in jcinit.c of IJG Jpeg-6a library.
	 */
	//=============================================================================
	private void jinitCompressMaster() throws IOException {
	    System.out.println("jinitCompressMaster");
	    // Initialize master control (includes parameter checking/processing).
	    master = new JPEGMaster(this, false);		// Full compression.
	    
	    // Preprocessing.
	    if ( !rawDataIn ) {
		cconvert = new JPEGColor(this);
		downsample = new JPEGSample(this);
		prep = new JPEGPrepController(this, false);	// Never need full buffer here.
	    }
	    
	    // Forward DCT.
	    //		jinit_forward_dct(cinfo);
	    
	    // Entropy encoding: either Huffman or arithmetic coding.
	    if ( arithCode) {
		throw new IOException(JPEGError.JERR_ARITH_NOTIMPL);
	    }
	    else {
		if ( progressiveMode ) {
		    //#ifdef C_PROGRESSIVE_SUPPORTED
		    //			jinit_phuff_encoder(cinfo);
		    //#else
		    throw new IOException(JPEGError.JERR_NOT_COMPILED);
		    //#endif
		}
		else {
		    //				jinit_huff_encoder(cinfo);
		}
	    }
	    
	    // Need a full-image coefficient buffer in any multi-pass mode.
	    //		jinit_c_coef_controller(cinfo,
	    //				(cinfo->num_scans > 1 || cinfo->optimize_coding));
	    main = new JPEGMainController(this, false);		// Never need full buffer here.
	    
	    //		jinit_marker_writer(cinfo);		// No need for this anymore. KF.
	    
	    // We can now tell the memory manager to allocate virtual arrays.
	    //		(*cinfo->mem->realize_virt_arrays) ((j_common_ptr) cinfo);
	    
	    // Write the datastream header (SOI) immediately.
	    // Frame and scan headers are postponed till later.
	    // This lets application insert special markers after the SOI.
	    writeFileHeader();
	}
	
}	// End of class.

/* Huffman coding tables */
class JpegHuffmanTbl {
    /* These two fields directly represent the contents of a JPEG DHT marker */
    public int[]	bits = new int[17];		/* bits[k] = # of symbols with codes of */
    /* length k bits; bits[0] is unused */
    public int[]	huffval = new int[256];	/* The symbols, in order of incr code length */
    
	/* This field is used only during compression.  It's initialized FALSE when
	 * the table is created, and set TRUE when it's been output to the file.
	 * You could suppress output of a table by setting this to TRUE.
	 * (See jpeg_suppress_tables for an example.)
	 */
    public boolean sentTable;		// TRUE when table has been output.
}

