/*****************************************************************************
 *
 *                                JPEGColor.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.util.Vector;

/**
 * This file is more or less a translation of the jccolor.c file
 * of the Independent JPEG Group's software.
 *
 * This file contains input colorspace conversion routines.
 *
 * @author Kary FR&Auml;MLING
 */
/*
#define JPEG_INTERNALS
#include "jinclude.h"
#include "jpeglib.h"
 */

import java.io.IOException;

public class JPEGColor {
    
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    
    public static final int		RGB_RED = 0;		/* Offset of Red in an RGB scanline element */
    public static final int		RGB_GREEN = 1;		/* Offset of Green */
    public static final int		RGB_BLUE = 2;		/* Offset of Blue */
    public static final int		RGB_RED_MASK = 0xFF0000;	// Mask for getting red value of int-coded RGB.
    public static final int		RGB_GREEN_MASK = 0xFF00;	// Mask for getting green value of int-coded RGB.
    public static final int		RGB_BLUE_MASK = 0xFF;	// Mask for getting blue value of int-coded RGB.
    public static final int		RGB_PIXELSIZE = 3;	// JSAMPLEs per RGB scanline element.
    public static final int		MAXJSAMPLE = 255;	// Valid maximum value for 8-bit samples.
    public static final int		CENTERJSAMPLE= 128;
    
    /**
     * Color conversion modes.
     */
    public static final int		NULL_CONVERT = 0;
    public static final int		GRAYSCALE_CONVERT = 1;
    public static final int		RGB_GRAY_CONVERT = 2;
    public static final int		RGB_YCC_CONVERT = 3;
    public static final int		CMYK_YCCK_CONVERT = 4;
    
    /**
     * Color treatment pass indicators.
     */
    public static final int		NULL_METHOD = 0;
    public static final int		RGB_YCC_START = 1;
    
    //--------------------------------------------------------------------------------------
    // Public fields.
    //--------------------------------------------------------------------------------------
    public int	colorConvert;
    public int	startPass;
    
    //--------------------------------------------------------------------------------------
    // Protected constants.
    //--------------------------------------------------------------------------------------
    /**
     * Known color spaces.
     */
    protected static final int JCS_UNKNOWN = 0;			// error/unspecified.
    protected static final int JCS_GRAYSCALE = 1;		// monochrome.
    protected static final int JCS_RGB = 2;				// red/green/blue.
    protected static final int JCS_YCbCr = 3;			// Y/Cb/Cr (also known as YUV).
    protected static final int JCS_CMYK = 4;			// C/M/Y/K.
    protected static final int JCS_YCCK = 5;			// Y/Cb/Cr/K.
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private static final int TABLE_SIZE = 8*(MAXJSAMPLE+1);
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private int[]	rgbYccTab;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Module initialization routine for input colorspace conversion.
     *
     * See jinit_color_converter in jccolor.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public JPEGColor(JPEGImageSaver cinfo) throws IOException {
	// Set start_pass to null method until we find out differently.
	startPass = NULL_METHOD;
	
	// Make sure input_components agrees with in_color_space.
	switch ( cinfo.inColorSpace) {
	    case JCS_GRAYSCALE:
		if ( cinfo.inputComponents != 1 )
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		break;
		
	    case JCS_RGB:
		//		#if RGB_PIXELSIZE != 3
		//			if (cinfo.inputComponents != RGB_PIXELSIZE ) ;
		//				throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		//			break;
		//		#endif // else share code with YCbCr.
		
	    case JCS_YCbCr:
		if ( cinfo.inputComponents != 3 )
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		break;
		
	    case JCS_CMYK:
	    case JCS_YCCK:
		if ( cinfo.inputComponents != 4 )
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		break;
		
	    default:			// JCS_UNKNOWN can be anything.
		if ( cinfo.inputComponents < 1 )
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		break;
	}
	
	// Check num_components, set conversion method based on requested space.
	switch ( cinfo.jpegColorSpace ) {
	    case JCS_GRAYSCALE:
		if ( cinfo.inputComponents != 1)
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		if ( cinfo.inColorSpace == JCS_GRAYSCALE)
		    colorConvert = GRAYSCALE_CONVERT;
		else if ( cinfo.inColorSpace == JCS_RGB) {
		    startPass = RGB_YCC_START;
		    colorConvert = RGB_GRAY_CONVERT;
		} else if ( cinfo.inColorSpace == JCS_YCbCr)
		    colorConvert = GRAYSCALE_CONVERT;
		else
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		break;
		
	    case JCS_RGB:
		if ( cinfo.inputComponents != 3)
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		if ( cinfo.inColorSpace == JCS_RGB && RGB_PIXELSIZE == 3)
		    colorConvert = NULL_CONVERT;
		else
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		break;
		
	    case JCS_YCbCr:
		if ( cinfo.inputComponents != 3)
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		if ( cinfo.inColorSpace == JCS_RGB) {
		    startPass = RGB_YCC_START;
		    colorConvert = RGB_YCC_CONVERT;
		} else if ( cinfo.inColorSpace == JCS_YCbCr)
		    colorConvert = NULL_CONVERT;
		else
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		break;
		
	    case JCS_CMYK:
		if ( cinfo.inputComponents != 4)
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		if ( cinfo.inColorSpace == JCS_CMYK)
		    colorConvert = NULL_CONVERT;
		else
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		break;
		
	    case JCS_YCCK:
		if ( cinfo.inputComponents != 4)
		    throw new IOException(JPEGError.JERR_BAD_IN_COLORSPACE);
		if ( cinfo.inColorSpace == JCS_CMYK) {
		    startPass = RGB_YCC_START;
		    colorConvert = CMYK_YCCK_CONVERT;
		} else if ( cinfo.inColorSpace == JCS_YCCK)
		    colorConvert = NULL_CONVERT;
		else
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		break;
		
	    default:			// allow null conversion of JCS_UNKNOWN.
		if ( cinfo.jpegColorSpace !=  cinfo.inColorSpace ||
		cinfo.numComponents != cinfo.inputComponents )
		    throw new IOException(JPEGError.JERR_CONVERSION_NOTIMPL);
		colorConvert = NULL_CONVERT;
		break;
	}
    }
    
    //=============================================================================
    // startPass
    //=============================================================================
    /**
     * Method that replaces the function pointes in the C JPEG implementation.
     */
    //=============================================================================
    public void startPass(JPEGImageSaver cinfo) throws IOException {
	switch ( startPass ) {
	    case NULL_METHOD:
		return;
	    case RGB_YCC_START:
		rgbYccStart(cinfo);
		break;
	    default:
		// Who cares...
	}
    }
    
    /**************** RGB -> YCbCr conversion: most common case **************/
    
	/*
	 * YCbCr is defined per CCIR 601-1, except that Cb and Cr are
	 * normalized to the range 0..MAXJSAMPLE rather than -0.5 .. 0.5.
	 * The conversion equations to be implemented are therefore
	 *	Y  =  0.29900 * R + 0.58700 * G + 0.11400 * B
	 *	Cb = -0.16874 * R - 0.33126 * G + 0.50000 * B  + CENTERJSAMPLE
	 *	Cr =  0.50000 * R - 0.41869 * G - 0.08131 * B  + CENTERJSAMPLE
	 * (These numbers are derived from TIFF 6.0 section 21, dated 3-June-92.)
	 * Note: older versions of the IJG code used a zero offset of MAXJSAMPLE/2,
	 * rather than CENTERJSAMPLE, for Cb and Cr.  This gave equal positive and
	 * negative swings for Cb/Cr, but meant that grayscale values (Cb=Cr=0)
	 * were not represented exactly.  Now we sacrifice exact representation of
	 * maximum red and maximum blue in order to get exact grayscales.
	 *
	 * To avoid floating-point arithmetic, we represent the fractional constants
	 * as integers scaled up by 2^16 (about 4 digits precision); we have to divide
	 * the products by 2^16, with appropriate rounding, to get the correct answer.
	 *
	 * For even more speed, we avoid doing any multiplications in the inner loop
	 * by precalculating the constants times R,G,B for all possible values.
	 * For 8-bit JSAMPLEs this is very reasonable (only 256 entries per table);
	 * for 12-bit samples it is still acceptable.  It's not very reasonable for
	 * 16-bit samples, but if you want lossless storage you shouldn't be changing
	 * colorspace anyway.
	 * The CENTERJSAMPLE offsets and the rounding fudge-factor of 0.5 are included
	 * in the tables to save adding them separately in the inner loop.
	 */
    
    private final int	SCALEBITS = 16;		// Speediest right-shift on some machines.
    private final int	CBCR_OFFSET = (CENTERJSAMPLE << SCALEBITS);
    private final int	ONE_HALF = (1 << (SCALEBITS-1));
    //	#define FIX(x)		((INT32) ((x) * (1L<<SCALEBITS) + 0.5))
    
	/* We allocate one big table and divide it up into eight parts, instead of
	 * doing eight alloc_small requests.  This lets us use a single table base
	 * address, which can be held in a register in the inner loops on many
	 * machines (more than can hold all eight addresses, anyway).
	 */
    private static final int R_Y_OFF = 0;					// offset to R => Y section.
    private static final int G_Y_OFF = (1*(MAXJSAMPLE+1));	// offset to G => Y section.
    private static final int B_Y_OFF = (2*(MAXJSAMPLE+1));	// etc.
    private static final int R_CB_OFF = (3*(MAXJSAMPLE+1));
    private static final int G_CB_OFF = (4*(MAXJSAMPLE+1));
    private static final int B_CB_OFF = (5*(MAXJSAMPLE+1));
    private static final int R_CR_OFF = B_CB_OFF;			// B=>Cb, R=>Cr are the same.
    private static final int G_CR_OFF = (6*(MAXJSAMPLE+1));
    private static final int B_CR_OFF = (7*(MAXJSAMPLE+1));
    
    //=============================================================================
    // rgbYccStart
    //=============================================================================
    /**
     * Initialize for RGB->YCC colorspace conversion.
     * We pre-calculate all the possible values, so that we only need to do a
     * table lookup afterwards.
     *
     * See rgb_ycc_start in jccolor.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void rgbYccStart(JPEGImageSaver cinfo) throws IOException {
	int		i;
	
	// Allocate and fill in the conversion tables.
	rgbYccTab = new int[TABLE_SIZE];
	
	for ( i = 0 ; i <= MAXJSAMPLE ; i++) {
	    rgbYccTab[i + R_Y_OFF] = ((int) (0.29900*(1<<SCALEBITS) + 0.5))*i;	// FIX(0.29900)*i;
	    rgbYccTab[i + G_Y_OFF] = ((int) (0.58700*(1<<SCALEBITS) + 0.5))*i;	// FIX(0.58700)*i;
	    rgbYccTab[i + B_Y_OFF] = ((int) (0.11400*(1<<SCALEBITS) + 0.5))*i + ONE_HALF;	// FIX(0.11400)*i + ONE_HALF;
	    rgbYccTab[i + R_CB_OFF] = -((int) (0.16874*(1<<SCALEBITS) + 0.5))*i;	// (-FIX(0.16874))*i;
	    rgbYccTab[i + G_CB_OFF] = -((int) (0.33126*(1<<SCALEBITS) + 0.5))*i;	//(-FIX(0.33126))*i;
	    
	    // We use a rounding fudge-factor of 0.5-epsilon for Cb and Cr.
	    // This ensures that the maximum output will round to MAXJSAMPLE
	    // not MAXJSAMPLE+1, and thus that we don't have to range-limit.
	    rgbYccTab[i+B_CB_OFF] = ((int) (0.50000*(1<<SCALEBITS) + 0.5))*i + CBCR_OFFSET + ONE_HALF - 1;	// FIX(0.50000)*i + CBCR_OFFSET + ONE_HALF-1;
	    //  B=>Cb and R=>Cr tables are the same
	    //    rgbYccTab[i+R_CR_OFF] = FIX(0.50000) * i    + CBCR_OFFSET + ONE_HALF-1;
	    
	    rgbYccTab[i + G_CR_OFF] = -((int) (0.41869*(1<<SCALEBITS) + 0.5))*i;	//(-FIX(0.41869))*i;
	    rgbYccTab[i + B_CR_OFF] = -((int) (0.08131*(1<<SCALEBITS) + 0.5))*i;	//(-FIX(0.08131))*i;
	}
    }
    
    //=============================================================================
    // colorConvert
    //=============================================================================
    /**
     * Replaces the color_convert function pointer in the original C version.
     */
    //=============================================================================
    public void colorConvert(JPEGImageSaver cinfo, int[] inputBuf, Vector outputBuf,
    int outputRow, int numRows, int off, int scansize) throws IOException {
	switch ( colorConvert ) {
	    case NULL_CONVERT:
		break;
	    case GRAYSCALE_CONVERT:
		break;
	    case RGB_GRAY_CONVERT:
		break;
	    case RGB_YCC_CONVERT:
		// This is the only one we care about for the moment.
		rgbYccConvert(cinfo, inputBuf, outputBuf,
		outputRow, numRows, off, scansize);
		break;
	    case CMYK_YCCK_CONVERT:
		break;
	    default:
		// Should never happen, unless we have a really big bug. KF.
		break;
	}
    }
    
    //=============================================================================
    // rgbYccConvert
    //=============================================================================
    /**
     * Convert some rows of samples to the JPEG colorspace.
     *
     * Note that we change from the application's interleaved-pixel format
     * to our internal noninterleaved, one-plane-per-component format.
     * The input buffer is therefore three times as wide as the output buffer.
     *
     * A starting row offset is provided only for the output buffer.  The caller
     * can easily adjust the passed input_buf value to accommodate any row
     * offset required on that side.
     *
     * See rgb_ycc_convert in jccolor.c of IJG Jpeg-6a library.
     */
    //=============================================================================
    public void rgbYccConvert(JPEGImageSaver cinfo, int[] inputBuf, Vector outputBuf,
    int outputRow, int numRows, int off, int scansize) throws IOException {
	int		r, g, b;
	int[]	ctab = rgbYccTab;
	int		inptr;
	//		JSAMPROW inptr;
	int[][]	comp1out, comp2out, comp3out;
	int		col;
	int		num_cols = cinfo.width;
	
	comp1out = (int[][]) outputBuf.elementAt(0);
	comp2out = (int[][]) outputBuf.elementAt(1);
	comp3out = (int[][]) outputBuf.elementAt(2);
	
	// Changed from original C version in order to avoid using a new buffer
	// after the ImageConsumer setPixels() call.
	inptr = off;
	while ( --numRows >= 0 ) {
	    for ( col = 0 ; col < num_cols ; col++) {
		// This changes from the original C version since Java RGB values
		// are coded as int. KF.
		r = (inputBuf[inptr]&RGB_RED_MASK)>>16;
		g = (inputBuf[inptr]&RGB_GREEN_MASK)>>8;
		b = inputBuf[inptr]&RGB_BLUE_MASK;
		inptr++;
		
		// If the inputs are 0..MAXJSAMPLE, the outputs of these equations
		// must be too; we do not need an explicit range-limiting operation.
		// Hence the value being shifted is never negative, and we don't
		// need the general RIGHT_SHIFT macro.
		// *** OK, we leave it like this even though it wastes some memory. KF.
		
		//* Y *
		comp1out[outputRow][col] = (int) ((ctab[r + R_Y_OFF] + ctab[g + G_Y_OFF] + ctab[b + B_Y_OFF])>> SCALEBITS);
		
		//* Cb *
		comp2out[outputRow][col] = (int) ((ctab[r + R_CB_OFF] + ctab[g + G_CB_OFF] + ctab[b + B_CB_OFF])>> SCALEBITS);
		
		//* Cr *
		comp3out[outputRow][col] = (int) ((ctab[r + R_CR_OFF] + ctab[g + G_CR_OFF] + ctab[b + B_CR_OFF])>> SCALEBITS);
		// if ( col == 0 ) System.out.println("y:" + comp1out[outputRow][col] + ", Cb:" +
		//		comp2out[outputRow][col] + ", Cr:" + comp3out[outputRow][col]);
	    }
	    outputRow++;
	}
    }
    
    /**************** Cases other than RGB -> YCbCr **************/
    
    
/*
 * Convert some rows of samples to the JPEG colorspace.
 * This version handles RGB->grayscale conversion, which is the same
 * as the RGB->Y portion of RGB->YCbCr.
 * We assume rgb_ycc_start has been called (we only use the Y tables).
 */
/*
METHODDEF(void)
rgb_gray_convert (j_compress_ptr cinfo,
		  JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
		  JDIMENSION output_row, int num_rows)
{
  my_cconvert_ptr cconvert = (my_cconvert_ptr) cinfo->cconvert;
  register int r, g, b;
  register INT32 * ctab = cconvert->rgb_ycc_tab;
  register JSAMPROW inptr;
  register JSAMPROW outptr;
  register JDIMENSION col;
  JDIMENSION num_cols = cinfo->image_width;
 
  while (--num_rows >= 0) {
    inptr = *input_buf++;
    outptr = output_buf[0][output_row];
    output_row++;
    for (col = 0; col < num_cols; col++) {
      r = GETJSAMPLE(inptr[RGB_RED]);
      g = GETJSAMPLE(inptr[RGB_GREEN]);
      b = GETJSAMPLE(inptr[RGB_BLUE]);
      inptr += RGB_PIXELSIZE;
 
      //* Y *
      outptr[col] = (JSAMPLE)
		((ctab[r+R_Y_OFF] + ctab[g+G_Y_OFF] + ctab[b+B_Y_OFF])
		 >> SCALEBITS);
    }
  }
}
 */
    
/*
 * Convert some rows of samples to the JPEG colorspace.
 * This version handles Adobe-style CMYK->YCCK conversion,
 * where we convert R=1-C, G=1-M, and B=1-Y to YCbCr using the same
 * conversion as above, while passing K (black) unchanged.
 * We assume rgb_ycc_start has been called.
 */
/*
METHODDEF(void)
cmyk_ycck_convert (j_compress_ptr cinfo,
		   JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
		   JDIMENSION output_row, int num_rows)
{
  my_cconvert_ptr cconvert = (my_cconvert_ptr) cinfo->cconvert;
  register int r, g, b;
  register INT32 * ctab = cconvert->rgb_ycc_tab;
  register JSAMPROW inptr;
  register JSAMPROW outptr0, outptr1, outptr2, outptr3;
  register JDIMENSION col;
  JDIMENSION num_cols = cinfo->image_width;
 
  while (--num_rows >= 0) {
    inptr = *input_buf++;
    outptr0 = output_buf[0][output_row];
    outptr1 = output_buf[1][output_row];
    outptr2 = output_buf[2][output_row];
    outptr3 = output_buf[3][output_row];
    output_row++;
    for (col = 0; col < num_cols; col++) {
      r = MAXJSAMPLE - GETJSAMPLE(inptr[0]);
      g = MAXJSAMPLE - GETJSAMPLE(inptr[1]);
      b = MAXJSAMPLE - GETJSAMPLE(inptr[2]);
 
      // K passes through as-is.
      outptr3[col] = inptr[3];	// Don't need GETJSAMPLE here.
      inptr += 4;
      // If the inputs are 0..MAXJSAMPLE, the outputs of these equations
      // must be too; we do not need an explicit range-limiting operation.
      // Hence the value being shifted is never negative, and we don't
      // need the general RIGHT_SHIFT macro.
      //* Y *
      outptr0[col] = (JSAMPLE)
		((ctab[r+R_Y_OFF] + ctab[g+G_Y_OFF] + ctab[b+B_Y_OFF])
		 >> SCALEBITS);
      //* Cb *
      outptr1[col] = (JSAMPLE)
		((ctab[r+R_CB_OFF] + ctab[g+G_CB_OFF] + ctab[b+B_CB_OFF])
		 >> SCALEBITS);
      //* Cr *
      outptr2[col] = (JSAMPLE)
		((ctab[r+R_CR_OFF] + ctab[g+G_CR_OFF] + ctab[b+B_CR_OFF])
		 >> SCALEBITS);
    }
  }
}
 */
    
/*
 * Convert some rows of samples to the JPEG colorspace.
 * This version handles grayscale output with no conversion.
 * The source can be either plain grayscale or YCbCr (since Y == gray).
 */
/*
METHODDEF(void)
grayscale_convert (j_compress_ptr cinfo,
		   JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
		   JDIMENSION output_row, int num_rows)
{
  register JSAMPROW inptr;
  register JSAMPROW outptr;
  register JDIMENSION col;
  JDIMENSION num_cols = cinfo->image_width;
  int instride = cinfo->input_components;
 
  while (--num_rows >= 0) {
    inptr = *input_buf++;
    outptr = output_buf[0][output_row];
    output_row++;
    for (col = 0; col < num_cols; col++) {
      outptr[col] = inptr[0];	// Don't need GETJSAMPLE() here.
      inptr += instride;
    }
  }
}
 */
    
/*
 * Convert some rows of samples to the JPEG colorspace.
 * This version handles multi-component colorspaces without conversion.
 * We assume input_components == num_components.
 */
/*
METHODDEF(void)
null_convert (j_compress_ptr cinfo,
	      JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
	      JDIMENSION output_row, int num_rows)
{
  register JSAMPROW inptr;
  register JSAMPROW outptr;
  register JDIMENSION col;
  register int ci;
  int nc = cinfo->num_components;
  JDIMENSION num_cols = cinfo->image_width;
 
  while (--num_rows >= 0) {
    // It seems fastest to make a separate pass for each component.
    for (ci = 0; ci < nc; ci++) {
      inptr = *input_buf;
      outptr = output_buf[ci][output_row];
      for (col = 0; col < num_cols; col++) {
	outptr[col] = inptr[ci]; // Don't need GETJSAMPLE() here.
	inptr += nc;
      }
    }
    input_buf++;
    output_row++;
  }
}
 */
    
/*
 * Empty method for start_pass.
 */
/*
METHODDEF(void)
null_method (j_compress_ptr cinfo)
{
  // No work needed.
}
 */
    
} // End of class.


