/*****************************************************************************
 *
 *                           ImageSerializer.java
 *
 * Class for saving AWT images as serialized objects.
 * This requires saving the pixels, but eventually other information too:
 *   - Always store the image dimension.
 *   - Color model parameters for indexed color models.
 *   - Properties if any (not implemented yet).
 *
 * Created by Kary FRAMLING 24/4/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.*;
import java.awt.image.*;
import java.util.zip.*;

public class ImageSerializer extends ImageSaverInterface
implements Serializable {
    
    //-----------------------------------------------------------------
    // Public constants.
    //-----------------------------------------------------------------
    public static final int		NO_COMPRESSION = 0;
    public static final String	FORMAT_CODE = "JSI";
    public static final String	FORMAT_COMPLETE_NAME = "Java Serialized Image";
    public static final String	FORMAT_EXTENSION = "jsi";
    
    //-----------------------------------------------------------------
    // Private variables.
    //-----------------------------------------------------------------
    private ColorModel	imgColorModel;
    private Rectangle	imgBounds;
    private byte[]		imgPixels;
    private int			compression;
    private int			saveStatus;
    
    //-----------------------------------------------------------------
    // main
    //-----------------------------------------------------------------
    /**
     * Test main function.
     */
    //-----------------------------------------------------------------
/*	public static void main(String[] argv)
	{
		ImageSerializer	is;
		Image			img;
 
		if ( new File(argv[0]).exists() ) {
			img = Toolkit.getDefaultToolkit().getImage(argv[0]);
			is = new ImageSerializer(img);
 
			try {
				FileOutputStream ostream = new FileOutputStream("pixs.bin");
				GZIPOutputStream gzostream = new GZIPOutputStream(ostream);
				ObjectOutputStream p = new ObjectOutputStream(gzostream);
				p.writeObject(is);
				p.close();
				gzostream.close();
				ostream.close();
			} catch ( IOException e ) { System.out.println(e); }
			System.out.println("Finished writing file!");
 
			ImageSerializer is_read = new ImageSerializer();
			try {
				FileInputStream istream = new FileInputStream("pixs.bin");
				GZIPInputStream gzistream = new GZIPInputStream(istream);
				ObjectInputStream p = new ObjectInputStream(gzistream);
				is_read = (ImageSerializer) p.readObject();
				p.close();
				gzistream.close();
				istream.close();
			}
			catch ( IOException e ) { System.err.println(e); }
			catch ( ClassNotFoundException e ) { System.err.println(e); }
 
			ImageFrame f = new ImageFrame(is_read.getImage());
			f.setSize(550, 500);
			f.show();
		}
//		System.exit(0);
	}
 */
    //-----------------------------------------------------------------
    // Constructor
    //-----------------------------------------------------------------
    /**
     * Default constructor. Use this when you want to read the
     * serialized image.
     */
    //-----------------------------------------------------------------
    public ImageSerializer() {
	compression = NO_COMPRESSION;
    }
    
    //-----------------------------------------------------------------
    // Constructor
    //-----------------------------------------------------------------
    /**
     * Use this constructor when you want to serialize an image
     * onto disk.
     */
    //-----------------------------------------------------------------
    public ImageSerializer(Image img) {
	this();
	saveImage = img;
    }
    
    //=============================================================================
    /**
     * ImageSaverInterface method implementations.
     *
     * @author Kary FR&Auml;MLING 12/2/1998.
     */
    //=============================================================================
    public String getFormatCode() { return FORMAT_CODE; }
    public String getFormatString() { return FORMAT_COMPLETE_NAME; }
    public String getFormatExtension() { return FORMAT_EXTENSION; }
    
    public boolean saveIt() { return true; }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return saveStatus;
    }
    
    //-----------------------------------------------------------------
    // getImage
    //-----------------------------------------------------------------
    /**
     * Get the image.
     */
    //-----------------------------------------------------------------
    public Image getImage() {
	return saveImage;
    }
    
    //-----------------------------------------------------------------
    // setImage
    //-----------------------------------------------------------------
    /**
     * Set the image to serialize.
     */
    //-----------------------------------------------------------------
    public void setImage(Image img) {
	saveImage = img;
    }
    
    //-----------------------------------------------------------------
    // getCompression
    //-----------------------------------------------------------------
    /**
     * Get the compression used.
     */
    //-----------------------------------------------------------------
    public int getCompression() {
	return compression;
    }
    
    //-----------------------------------------------------------------
    // setCompression
    //-----------------------------------------------------------------
    /**
     * Set the compression used.
     */
    //-----------------------------------------------------------------
    public void setCompression(int c) {
	compression = c;
    }
    
    //-----------------------------------------------------------------
    // getColorModel
    //-----------------------------------------------------------------
    /**
     * Get the color model of the image. This is known after serializing
     * the image or after reading it from disk.
     */
    //-----------------------------------------------------------------
    public ColorModel getColorModel() {
	return imgColorModel;
    }
    
    //-----------------------------------------------------------------
    // getImageWidth
    //-----------------------------------------------------------------
    /**
     * Get the width of the image. This is known after serializing
     * the image or after reading it from disk.
     */
    //-----------------------------------------------------------------
    public int getImageWidth() {
	return ( imgBounds == null ) ? -1 : imgBounds.width;
    }
    
    //-----------------------------------------------------------------
    // getImageHeight
    //-----------------------------------------------------------------
    /**
     * Get the height of the image. This is known after serializing
     * the image or after reading it from disk.
     */
    //-----------------------------------------------------------------
    public int getImageHeight() {
	return ( imgBounds == null ) ? -1 : imgBounds.height;
    }
    
    //-----------------------------------------------------------------
    // writeObject
    //-----------------------------------------------------------------
    /**
     * Write the Image as a serialized object.
     */
    //-----------------------------------------------------------------
    private void writeObject(ObjectOutputStream stream)
    throws IOException {
	// Elementary verifications.
	if ( saveImage == null ) return;
	
	// Grab image pixels. This might be optimized in future versions for
	// grabbing only an area of the image;
	PixelGrabber pg = new PixelGrabber(saveImage, 0, 0, -1, -1, false);
	try {
	    pg.grabPixels();
	} catch (InterruptedException e) {
	    System.err.println("Interrupted waiting for pixels!");
	    return;
	}
	if ((pg.getStatus() & ImageObserver.ABORT) != 0) {
	    System.err.println("Image fetch aborted or errored.");
	    return;
	}
	
	// Write out compression method used.
	stream.writeInt(compression);
	
	// Write out image bounds.
	imgBounds = new Rectangle(0, 0, pg.getWidth(), pg.getHeight());
	stream.writeObject(imgBounds);
	
	// Write out the color model parameters if it is an indexed color
	// model. If it is a direct color model, then we write out the pixel size
	// as an integer object
	imgColorModel = pg.getColorModel();
	if ( imgColorModel instanceof IndexColorModel ) {
	    stream.writeObject(new IndexColorModelSerializer((IndexColorModel) imgColorModel));
	}
	else if ( imgColorModel instanceof DirectColorModel ) {
	    stream.writeObject(new DirectColorModelSerializer((DirectColorModel) imgColorModel));
	}
	
	// Then write out the pixel array object returned. We suppose it is a
	// serializable object, which should be the case. It is so both for byte[]
	// and int[].
	Object pixel_array = pg.getPixels();
	stream.writeObject(pixel_array);
    }
    
    //-----------------------------------------------------------------
    // readObject
    //-----------------------------------------------------------------
    /**
     * Read the Image as a serialized object.
     */
    //-----------------------------------------------------------------
    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException {
	Object color_model;
	Object pixel_array;
	
	// Write out compression method used.
	compression = stream.readInt();
	
	// Get image bounds.
	imgBounds = (Rectangle) stream.readObject();
	
	// Read the color model parameters if it is an indexed color
	// model.
	color_model = stream.readObject();
	if ( color_model instanceof IndexColorModelSerializer ) {
	    imgColorModel = ((IndexColorModelSerializer) color_model).getModel();
	}
	else if ( color_model instanceof DirectColorModelSerializer ) {
	    imgColorModel = ((DirectColorModelSerializer) color_model).getModel();
	}
	else {
	    throw new IOException("No ColorModel!");
	}
	
	// Then read in the pixels and construct the image from the information
	// that we have.
	pixel_array = stream.readObject();
	if ( pixel_array instanceof byte[] ) {
	    byte[] byte_pix_array = (byte[]) pixel_array;
	    saveImage = Toolkit.getDefaultToolkit().createImage(
	    new MemoryImageSource(imgBounds.width, imgBounds.height,
	    imgColorModel, byte_pix_array, 0, imgBounds.width));
	}
	else if ( pixel_array instanceof int[] ) {
	    int[] int_pix_array = (int[]) pixel_array;
	    saveImage = Toolkit.getDefaultToolkit().createImage(
	    new MemoryImageSource(imgBounds.width, imgBounds.height,
	    imgColorModel, int_pix_array, 0, imgBounds.width));
	}
	else {
	    throw new IOException("Invalid pixel format!");
	}
    }
}

