/*****************************************************************************
 *
 *                             ImageSaverObject.java
 *
 * Class that saves gives methods to save Image objects in different formats.
 *
 * Java source created by Kary FRAMLING 12/2/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.*;
import java.awt.image.ImageObserver;
import java.io.Serializable;

public class ImageSaverObject implements Serializable {
    
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final String	STR_SAVE_IMAGE_DIAL_TITLE = "Save image...";
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private transient Image		saveImage;
    private transient String	savePath;
    private String 				saveFormat;
    private transient ImageSaverInterface	imageSaver;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    public ImageSaverObject() {
    }
    
    //=============================================================================
    // saveIt
    //=============================================================================
    /**
     * Save the image into a graphics file of the given format. This
     * static method is a convenience method which makes it possible
     * to save an image with just one call.	Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 12/2/1998.
     */
    //=============================================================================
    public boolean saveIt() {
	// Verify that we have all necessary information.
	if ( saveImage == null || savePath == null || savePath.equals("") || saveFormat == null )
	    return false;
	
	// Verify that we have a supported format.
	if ( saveFormat.equals(TIFFImageSaver.FORMAT_CODE) ) {
	    imageSaver = new TIFFImageSaver();
	}
	else if ( saveFormat.equals(TargaImageSaver.FORMAT_CODE) ) {
	    imageSaver = new TargaImageSaver();
	}
	else if ( saveFormat.equals(PICTImageSaver.FORMAT_CODE) ) {
	    imageSaver = new PICTImageSaver();
	}
	else if ( saveFormat.equals(BMPImageSaver.FORMAT_CODE) ) {
	    imageSaver = new BMPImageSaver();
	}
	else if ( saveFormat.equals(PNGImageSaver.FORMAT_CODE) ) {
	    imageSaver = new PNGImageSaver();
	}
	else {
	    return false;
	}
	
	// Everything should be OK, set up the imageSaver accordingly and save it.
	if ( imageSaver != null ) {
	    imageSaver.setSaveImage(saveImage);
	    imageSaver.setSavePath(savePath);
	    return imageSaver.saveIt();
	}
	
	// Some strange error occurred.
	return false;
    }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return ( imageSaver == null ) ? ImageObserver.ERROR : imageSaver.checkSave();
    }
    
    //=============================================================================
    // saveToFile
    //=============================================================================
    /**
     * Save the image into a graphics file of the given format. This
     * method is a convenience method which makes it possible
     * to save an image with just one call. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 12/2/1998.
     */
    //=============================================================================
    public boolean saveToFile(Image theImage, String thePath, String formatCode) {
	saveImage = theImage;
	savePath = thePath;
	saveFormat = formatCode;
	return saveIt();
    }
    
    //=============================================================================
    // setSaveImage
    //=============================================================================
    /**
     * Set the image to save.
     *
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    public void setSaveImage(Image img) {
	saveImage = img;
    }
    
    //=============================================================================
    // setSavePath
    //=============================================================================
    /**
     * Set the path where the file should be saved.
     *
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    public void setSavePath(String path) {
	savePath = path;
    }
    
    //=============================================================================
    // setSaveFormat
    //=============================================================================
    /**
     * Set the save format.
     *
     * @author Kary FR&Auml;ling 30/4/1998.
     */
    //=============================================================================
    public void setSaveFormat(String f) {
	saveFormat = f;
    }
    
    //=============================================================================
    // saveTIFF
    //=============================================================================
    /**
     * Save the image in TIFF format. "saveImage" and "savePath" have to be set
     * before. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 18/3/1998.
     */
    //=============================================================================
    public boolean saveTIFF() {
	// Save the image.
	imageSaver = new TIFFImageSaver();
	return saveWithImageSaver();
    }
    
    //=============================================================================
    // saveTarga
    //=============================================================================
    /**
     * Save the image in Targa format. "saveImage" and "savePath" have to be set
     * before. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 18/3/1998.
     */
    //=============================================================================
    public boolean saveTarga() {
	// Save the image.
	imageSaver = new TargaImageSaver();
	return saveWithImageSaver();
    }
    
    //=============================================================================
    // savePICT
    //=============================================================================
    /**
     * Save the image in PICT format. "saveImage" and "savePath" have to be set
     * before. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 1/6/1998.
     */
    //=============================================================================
    public boolean savePICT() {
	// Save the image.
	imageSaver = new PICTImageSaver();
	return saveWithImageSaver();
    }
    
    //=============================================================================
    // saveBMP
    //=============================================================================
    /**
     * Save the image in Windows BMP format. "saveImage" and "savePath" have to be set
     * before. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling 1/6/1998.
     */
    //=============================================================================
    public boolean saveBMP() {
	// Save the image.
	imageSaver = new BMPImageSaver();
	return saveWithImageSaver();
    }
    
    //=============================================================================
    // savePNG
    //=============================================================================
    /**
     * Save the image in PNG. "saveImage" and "savePath" have to be set
     * before. Return true if success, false otherwise.
     * REMARK! This is an asynchronous operation. Use checkSave() to know when
     * the saving is finished and how.
     *
     * @author Kary FR&Auml;ling
     */
    //=============================================================================
    public boolean savePNG() {
	// Save the image.
	imageSaver = new PNGImageSaver();
	return saveWithImageSaver();
    }
    
    //=============================================================================
    // saveAsTIFF
    //=============================================================================
    /**
     * Show the "Save As..." dialog and then save the image in TIFF format.
     *
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    public void saveAsTIFF() {
	// Get save path and check for cancel.
	if ( (savePath = getSaveAsPath()) == null ) return;
	
	// Save it.
	saveTIFF();
    }
    
    //=============================================================================
    // saveAsTarga
    //=============================================================================
    /**
     * Show the "Save As..." dialog and then save the image in Targa format.
     *
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    public void saveAsTarga() {
	// Get save path and check for cancel.
	if ( (savePath = getSaveAsPath()) == null ) return;
	
	// Save it.
	saveTarga();
    }
    
    //=============================================================================
    // saveAsPICT
    //=============================================================================
    /**
     * Show the "Save As..." dialog and then save the image in PICT format.
     *
     * @author Kary FR&Auml;ling 1/6/1998.
     */
    //=============================================================================
    public void saveAsPICT() {
	// Get save path and check for cancel.
	if ( (savePath = getSaveAsPath()) == null ) return;
	
	// Save it.
	savePICT();
    }
    
    //=============================================================================
    // saveAsBMP
    //=============================================================================
    /**
     * Show the "Save As..." dialog and then save the image in Windows BMP format.
     *
     * @author Kary FR&Auml;ling 1/6/1998.
     */
    //=============================================================================
    public void saveAsBMP() {
	// Get save path and check for cancel.
	if ( (savePath = getSaveAsPath()) == null ) return;
	
	// Save it.
	saveBMP();
    }
    
    //=============================================================================
    // saveAsPNG
    //=============================================================================
    /**
     * Show the "Save As..." dialog and then save the image in PNG format.
     *
     * @author Kary FR&Auml;ling
     */
    //=============================================================================
    public void saveAsPNG() {
	// Get save path and check for cancel.
	if ( (savePath = getSaveAsPath()) == null ) return;
	
	// Save it.
	savePNG();
    }
    
    //--------------------------------------------------------------------------------------
    // Private methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // getSaveAsPath
    //=============================================================================
    /**
     * Show the "Save As..." dialog and return the file path or null if cancel.
     *
     * @author Kary FR&Auml;ling 17/3/1998.
     */
    //=============================================================================
    private String getSaveAsPath() {
	// Check that we have a save image.
	if ( saveImage == null ) return null;
	
	// Show Save As... dialog.
	FileDialog the_dialog = new FileDialog(new Frame(), STR_SAVE_IMAGE_DIAL_TITLE, FileDialog.SAVE);
	the_dialog.show();
	
	// Check for cancel.
	if ( the_dialog.getFile() == null ) return null;
	
	// Return file path.
	return the_dialog.getDirectory() + the_dialog.getFile();
    }
    
    //=============================================================================
    // saveWithImageSaver
    //=============================================================================
    /**
     * Set up the current image saver object with the current parameters and
     * save the image.
     *
     * @author Kary FR&Auml;ling
     */
    //=============================================================================
    public boolean saveWithImageSaver() {
	// Check that we have a save image and a save path.
	if ( imageSaver == null || saveImage == null || savePath == null || savePath.equals("") ) return false;
	
	// Save the image.
	imageSaver.setSaveImage(saveImage);
	imageSaver.setSavePath(savePath);
	return imageSaver.saveIt();
    }
    
}


