/****************************************************************************
 *                               BMPImageSaver.java
 *
 * Java source created by Kary FRAMLING 29/5/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE, 
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jis;

import java.io.*;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ColorModel;
import java.util.Hashtable;
import java.awt.Toolkit;
import java.io.Serializable;

import fi.faidon.util.DataCompression;

/**
 * Class for saving an image in the Windows/OS2 Bitmap format.
 * For the moment, we just use an uncompressed, direct 24 bit format.
 *
 * @author Kary FR&Auml;MLING 29/5/1998
 */
public class BMPImageSaver extends ImageSaverInterface
implements Serializable, ImageConsumer {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final String	FORMAT_CODE = "BMP";
    public static final String	FORMAT_COMPLETE_NAME = "Windows/OS2 Bitmap";
    public static final String	FORMAT_EXTENSION = "bmp";
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final byte[]	TYPE_BMP_BUF = {0x42, 0x4D};	/* 'BM' */
    private final int		BITMAP_FILE_HDR_SIZE = 0xE;
    private final int		BITMAP_HDR_SIZE = 0x28;
    private final int		BITS_PER_PIXEL = 24;
    private final int		BYTES_PER_PIXEL = 3;
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private FileOutputStream	writeFileHandle;
    private int		width;
    private int		height;
    private int		saveStatus;
    private int		byteCount;
    private byte[]	pixBuf;
    private boolean	hdrWritten, imageWritten;
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    /**
     * ImageSaverInterface method implementations.
     */
    //=============================================================================
    public String getFormatCode() { return FORMAT_CODE; }
    public String getFormatString() { return FORMAT_COMPLETE_NAME; }
    public String getFormatExtension() { return FORMAT_EXTENSION; }
    
    //=============================================================================
    // saveIt
    //=============================================================================
    /**
     * Save the image.
     */
    //=============================================================================
    public boolean saveIt() {
	ImageProducer	ip;
	
	// Verify that we have an image.
	if ( saveImage == null ) return false;
	
	// No status information yet.
	saveStatus = 0;
	
	// Open the file.
	try {
	    writeFileHandle = new FileOutputStream(savePath);
	} catch ( IOException e ) { System.out.println("IOException occurred opening FileOutputStream : " + e); }
	
	// Return straight away if we couldn't get a handle.
	if ( writeFileHandle == null ) return false;
	
	// Image not written yet.
	hdrWritten = false;
	imageWritten = false;
	
	// Set us up as image consumer and start producing.
	ip = saveImage.getSource();
	if ( ip == null ) return false;
	ip.startProduction(this);
	
	// Nothing more to do, just get data and close file at the end.
	return true;
    }
    
    //=============================================================================
    // checkSave
    //=============================================================================
    /**
     * Return ImageObserver constants for indicating the state of the image saving.
     *
     * @author Kary FR&Auml;MLING 30/4/1998.
     */
    //=============================================================================
    public int checkSave() {
	return saveStatus;
    }
    
    //=============================================================================
    /**
     * ImageConsumer method implementations.
     */
    //=============================================================================
    public void setProperties(Hashtable props) {
	saveStatus |= ImageObserver.PROPERTIES;
    }
    
    public void setHints(int hintflags) {}
    public void setColorModel(ColorModel model) {}
    
    //=============================================================================
    // setDimensions
    //=============================================================================
    /**
     */
    //=============================================================================
    public void setDimensions(int w, int h) {
	int		i, bit_map_size;
	byte[]	buf;
	byte[]	byte_buf = new byte[1];
	byte[]	byte_buf_16 = new byte[2];
	byte[]	byte_buf_32 = new byte[4];
	
	// Store width and height.
	width = w;
	height = h;
	
	// Test if header is written. For some reason, this function sometimes gets called
	// twice, which is extremely annoying! So we test for it.
	if ( !hdrWritten ) {
	    
	    // No bytes written yet.
	    byteCount = 0;
	    
	    // One Bitmap image.
	    try { writeFileHandle.write(TYPE_BMP_BUF); } catch ( IOException e ) { System.out.println("Error!"); saveStatus = ImageObserver.ERROR; }
	    byteCount += TYPE_BMP_BUF.length;
	    
	    // Calculate the bit map size. This is known when no compression used.
	    bit_map_size = BYTES_PER_PIXEL*width*height;
	    
	    // File size. This is pretty annoying, since we cannot always know it
	    // from the start. If we use compression, then we probably have to come back
	    // in random access mode and set this information separately.
	    uintAsLittleEndianBytes(bit_map_size + BITMAP_FILE_HDR_SIZE + BITMAP_HDR_SIZE, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_32.length;
	    
	    // xHotspot and yHotspot.
	    uintAsLittleEndianBytes(0, byte_buf_16, 0, byte_buf_16.length);
	    try { writeFileHandle.write(byte_buf_16); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    try { writeFileHandle.write(byte_buf_16); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += 2*byte_buf_16.length;
	    
	    // offsetToBits.
	    uintAsLittleEndianBytes(BITMAP_FILE_HDR_SIZE + BITMAP_HDR_SIZE, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_32.length;
	    
	    // Bitmap header size. 0x28 because we only write out the minimal information
	    // needed.
	    uintAsLittleEndianBytes(BITMAP_HDR_SIZE, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_32.length;
	    
	    // Width and height. It would be attempting tu use left-down mode, but this
	    // seems to give problems for some programs.
	    uintAsLittleEndianBytes(width, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    uintAsLittleEndianBytes(height, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += 2*byte_buf_32.length;
	    
	    // numBitPlanes.
	    uintAsLittleEndianBytes(1, byte_buf_16, 0, byte_buf_16.length);
	    try { writeFileHandle.write(byte_buf_16); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_16.length;
	    
	    // numBitsPerPlane.
	    uintAsLittleEndianBytes(BITS_PER_PIXEL, byte_buf_16, 0, byte_buf_16.length);
	    try { writeFileHandle.write(byte_buf_16); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_16.length;
	    
	    // compressionScheme.
	    uintAsLittleEndianBytes(0, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_32.length;
	    
	    // sizeOfImageData.
	    uintAsLittleEndianBytes(bit_map_size, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += byte_buf_32.length;
	    
	    // xResolution, yResolution. No idea if this is really used.
	    uintAsLittleEndianBytes(0x2DB4, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += 2*byte_buf_32.length;
	    
	    // numColorsUsed, numImportantColors.
	    uintAsLittleEndianBytes(0, byte_buf_32, 0, byte_buf_32.length);
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    try { writeFileHandle.write(byte_buf_32); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += 2*byte_buf_32.length;
	    
	    // Set up the pixel buffer.
	    pixBuf = new byte[BYTES_PER_PIXEL*width*height];
	    
	    // Header is written. For some reason, this function sometimes gets called
	    // twice, which is extremely annoying! So we test for it.
	    hdrWritten = true;
	}
	
	// That's it. Now just pack the pixels into the file.
	
	// Update save status.
	saveStatus |= ImageObserver.WIDTH | ImageObserver.HEIGHT;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data. For this to work correctly,
     * pixels should be delivered in topdownleftright order with complete
     * scanlines. If we have several lines, the lines should be complete scanlines,
     * otherwise the saving fails.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, byte pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb;
	
	// Treat one line at a time.
	pix_byte_ind = y*BYTES_PER_PIXEL*width;
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		// Get RGB value.
		// Funny, we have to mask out the last byte for this to work correctly.
		rgb = model.getRGB(pixels[off + i*scansize + j]&0xFF);
		
		// Set red, green and blue components. These are stored in inverse
		// order in BMP files.
		pixBuf[pix_byte_ind++] = (byte) (rgb&0xFF);
		pixBuf[pix_byte_ind++] = (byte) ((rgb>>8)&0xFF);
		pixBuf[pix_byte_ind++] = (byte) ((rgb>>16)&0xFF);
	    }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // setPixels
    //=============================================================================
    /**
     * Write the pixels into the file as RGB data.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void setPixels(int x, int y, int w, int h,
    ColorModel model, int pixels[], int off,
    int scansize) {
	int		i, j, pix_byte_ind, rgb;
	
	// Treat one line at a time.
	pix_byte_ind = y*BYTES_PER_PIXEL*width;
	for ( i = 0 ; i < h ; i++ ) {
	    for ( j = 0 ; j < w ; j++ ) {
		// Get RGB value.
		rgb = model.getRGB(pixels[off + i*scansize + j]);
		
		// Set red, green and blue components. These are stored in inverse
		// order in BMP files.
		pixBuf[pix_byte_ind++] = (byte) (rgb&0xFF);
		pixBuf[pix_byte_ind++] = (byte) ((rgb>>8)&0xFF);
		pixBuf[pix_byte_ind++] = (byte) ((rgb>>16)&0xFF);
	    }
	}
	
	// Update save status.
	saveStatus |= ImageObserver.SOMEBITS;
    }
    
    //=============================================================================
    // imageComplete
    //=============================================================================
    /**
     * Get imageComplete message so that we can close the output file.
     * @see ImageConsumer.
     */
    //=============================================================================
    public void imageComplete(int status) {
	int		offset, padbytes;
	byte[]	pad_buf;
	
	// Strange tendency to get called twice here, so we test if the image has
	// already been written.
	if ( imageWritten ) return;
	
	// Pad first row to a 4-byte boundary. This is not needed here, since we have
	// selected a header size which makes us start at the right place.
	
	// Row bytes always have to be padded to 4-byte boundaries.
	padbytes = width%4;
	pad_buf = new byte[padbytes];
	
	// Write out the buffer in inversed order.
	for ( offset = (height - 1)*width*BYTES_PER_PIXEL ; offset >= 0 ; offset -= width*BYTES_PER_PIXEL ) {
	    try { writeFileHandle.write(pixBuf, offset, width*BYTES_PER_PIXEL); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
	    byteCount += width*BYTES_PER_PIXEL;
	    if ( padbytes != 0 ) {
		try { writeFileHandle.write(pad_buf); } catch ( IOException e ) { saveStatus = ImageObserver.ERROR; }
		byteCount += pad_buf.length;
	    }
	}
	
	// Close file
	try {
	    writeFileHandle.close();
	} catch ( IOException e ) {
	    saveStatus = ImageObserver.ERROR;
	}
	
	// We have written it, so no need to do it anymore.
	imageWritten = true;
	
	// Update save status.
	saveStatus |= ImageObserver.ALLBITS;
    }
    
    //--------------------------------------------------------------------------------------
    // Private methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // uintAsLittleEndianBytes
    //=============================================================================
    /**
     * Convert the int value into a little-endian byte array that starts from
     * index "startOff" and end at index "endOff" - 1 of "buf".
     */
    //=============================================================================
    private void uintAsLittleEndianBytes(int value, byte[] bytes, int startOff, int endOff) {
	int		i, shift_cnt;
	
	for ( i = startOff, shift_cnt = 0 ; i < endOff ; i++, shift_cnt += 8 ) {
	    bytes[i] = (byte) ((value>>shift_cnt)&0xFF);
	}
    }
    
    //=============================================================================
    // intAsLittleEndianBytes
    //=============================================================================
    /**
     * Return the int value given by the byte array in little-endian order.
     * Here we interpret the first bit of the first byte as being a sign bit.
     *
     * @author Kary FR&Auml;MLING 2/4/1998
     */
    //=============================================================================
    private void intAsLittleEndianBytes(int value, byte[] bytes, int startOffset, int endOffset) {
	int		abs_value;
	int		max_minus;
	
	// If it is negative, then do a special treatment. Otherwise just
	// the normal one.
	if ( value < 0 ) {
	    abs_value = -value;
	    max_minus = (int) Math.pow(2, (endOffset - startOffset)*8 - 1);
	    uintAsLittleEndianBytes(max_minus - abs_value, bytes, startOffset, endOffset);
	    bytes[endOffset - 1] |= 0x80;
	}
	else {
	    uintAsLittleEndianBytes(value, bytes, startOffset, endOffset);
	}
    }
    
}


