/*****************************************************************************
 *
 *                              ImageShowBean.java
 *
 * Bean for showing an image in a canvas.
 *
 * Created by Kary FRAMLING 16/3/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.gui;

import java.io.*;
import java.io.Serializable;
import java.awt.*;
import java.beans.*;

/**
 * Bean for showing an image in a canvas.
 *
 * @author Kary FR&Auml;MLING
 */
public class ImageShowBean extends Canvas implements Serializable {
    
    //--------------------------------------------------------------------------------------
    // Private constants.
    //--------------------------------------------------------------------------------------
    private final int	DEFAULT_WIDTH = 50;
    private final int	DEFAULT_HEIGHT = 50;
    
    //--------------------------------------------------------------------------------------
    // Public fields.
    //--------------------------------------------------------------------------------------
    
    //--------------------------------------------------------------------------------------
    // Private fields.
    //--------------------------------------------------------------------------------------
    private String	imagePath;
    private transient Image	image;
    private transient int	imageWidth;
    private transient int	imageHeight;
    private boolean	hasImageSize;
    private boolean	doIntermediateUpdates;
    private PropertyChangeSupport changes = new	PropertyChangeSupport(this);
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
    public static void main(String[] argv) {
	Frame f = new Frame("Test ImageShowBean");
	ImageShowBean ib = new ImageShowBean();
	ib.setImagePath("Testimg.gif");
	f.setLayout(new FlowLayout());
	f.add(ib);
	f.setSize(250, 200);
	f.show();
    }
    
    //=============================================================================
    // Constructor
    //=============================================================================
    /**
     * Initialize the component size so that we dont stay invisible.
     *
     * @author Kary FR&Auml;MLING 16/3/1998
     */
    //=============================================================================
    public ImageShowBean() {
	imageWidth = -1;
	imageHeight = -1;
	setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
	doIntermediateUpdates = true;
    }
    
    //=============================================================================
    // getImage
    //=============================================================================
    /**
     * Get the image.
     *
     * @author Kary FR&Auml;MLING 17/3/1998
     */
    //=============================================================================
    public Image getImage() { return image; }
    
    //=============================================================================
    // getImagePath
    //=============================================================================
    /**
     * Get the image path.
     *
     * @author Kary FR&Auml;MLING 16/3/1998
     */
    //=============================================================================
    public String getImagePath() { return imagePath; }
    
    //=============================================================================
    // setImagePath
    //=============================================================================
    /**
     * Set the image path and start preparing the image straight away.
     *
     * @author Kary FR&Auml;MLING 16/3/1998
     */
    //=============================================================================
    public void setImagePath(String path) {
	// See that we have a valid path.
	if ( path == null || path.equals("") ) return;
	
	// Get and prepare image.
	hasImageSize = false;
	imagePath = path;
	image = getToolkit().getImage(path);
	getToolkit().prepareImage(image, -1, -1, this);
	
	// Indicate to property listeners that the image has changed.
	changes.firePropertyChange("image", null, image);
    }
    
    //=============================================================================
    // getDoIntermediateUpdates
    //=============================================================================
    /**
     * Get intermediate paint mode.
     *
     * @author Kary FR&Auml;MLING 16/3/1998
     */
    //=============================================================================
    public boolean getDoIntermediateUpdates() { return doIntermediateUpdates; }
    
    //=============================================================================
    // setDoIntermediateUpdates
    //=============================================================================
    /**
     * Set intermediate paint mode. If true, we do progressive painting, if false
     * we draw the image only at the end.
     *
     * @author Kary FR&Auml;MLING 16/3/1998
     */
    //=============================================================================
    public void setDoIntermediateUpdates(boolean value) {
	doIntermediateUpdates = value;
    }
    
    //=============================================================================
    // paint
    //=============================================================================
    /**
     * Repaint the canvas. We also check when we know the width and the height
     * for the first time and resize ourselves accordingly. The parent container
     * is also notified about the change.
     *
     * @author Kary FR&Auml;MLING 2/7/1997
     */
    //=============================================================================
    public void paint(Graphics g) {
	Dimension	d;
	
	// See if we know the image size by now.
	if ( image != null && imageWidth == -1 ) {
	    imageWidth = image.getWidth(this);
	}
	if ( image != null && imageHeight == -1 ) {
	    imageHeight = image.getHeight(this);
	}
	
	// If we don't know the image width and height, then draw a default
	// image.
	if ( image == null || imageWidth == -1 || imageHeight == -1 ) {
	    d = getSize();
	    g.drawRect(0, 0, d.width - 1, d.height - 1);
	    g.drawLine(0, 0, d.width - 1, d.height - 1);
	    g.drawLine(0, d.height - 1, d.width - 1, 0);
	}
	else {
	    // We have the size, set our component size too if it hasn't been done yet.
	    if ( !hasImageSize ) {
		// Set the size and memorize that we have done it.
		setSize(imageWidth, imageHeight);
		hasImageSize = true;
		
		// Ask the parent container to refresh its layout.
		getParent().doLayout();
	    }
	    
	    // Paint the image. For the moment we do all image updates, partial ones too.
	    g.drawImage(image, 0, 0, this);
	}
    }
    
    //-----------------------------------------------------------------
    /**
     * PropertyChangeSupport methods.
     */
    //-----------------------------------------------------------------
    public void addPropertyChangeListener(PropertyChangeListener l) {
	changes.addPropertyChangeListener(l);
    }
    
    public void removePropertyChangeListener(PropertyChangeListener l) {
	changes.removePropertyChangeListener(l);
    }
    
}

