### This file is part of MINPANTU
### See file COPYRIGHT for pertinent copyright notices.

# Minor configuration that doesn't appear in the C-code.

set player0_color red2
set player1_color SteelBlue2
set hit_color green1
set ball_radius 5
set hit_distance 30

# Default red player is human and blue player the computer at level 3.
set red_player -2
set blue_player 3

set max_computer_effort 10

# Some naming.
set human_player -2


# Create the main window.

wm title . "minpantu"


### Create menus.

frame .menu -relief raised -borderwidth 1
pack .menu -side top -fill x

menubutton .menu.file -text "File" -menu .menu.file.m -underline 0
menu .menu.file.m
.menu.file.m add command -label "New" -command "new_game" -underline 0
.menu.file.m add command -label "Load" \
	-command "history_load game.mpt ; step_rewind" -underline 0
.menu.file.m add command -label "Save" \
	-command "history_save game.mpt" -underline 0
.menu.file.m add command -label "Quit" -command "destroy ." -underline 0

menubutton .menu.red -text "Red Player" -menu .menu.red.m  -underline 0
menu .menu.red.m
.menu.red.m add radio -label "Human" \
	-variable red_player -value -2 -command update_inputs
.menu.red.m add radio -label "Nobody" \
	-variable red_player -value -1 -command update_inputs
.menu.red.m add radio -label "Random" \
	-variable red_player -value 0 -command update_inputs
for {set i 1} {$i <= $max_computer_effort} {incr i} {
    .menu.red.m add radio -label "Computer Level $i" \
	    -variable red_player -value $i -command update_inputs
}

menubutton .menu.blue -text "Blue Player" -menu .menu.blue.m -underline 0
menu .menu.blue.m
.menu.blue.m add radio -label "Human" \
	-variable blue_player -value -2 -command update_inputs
.menu.blue.m add radio -label "Nobody" \
	-variable blue_player -value -1 -command update_inputs
.menu.blue.m add radio -label "Random" \
	-variable blue_player -value 0 -command update_inputs
for {set i 1} {$i <= $max_computer_effort} {incr i} {
    .menu.blue.m add radio -label "Computer Level $i" \
	    -variable blue_player -value $i -command update_inputs
}

menubutton .menu.help -text "Help" -menu .menu.help.m -underline 0
menu .menu.help.m
.menu.help.m add command -label "What's \"minpantu\"?" -command "name_help"
.menu.help.m add command -label "Rules of Minpantu?" -command "rule_help"
.menu.help.m add command -label "How to play?" -command "play_help"

pack .menu.file .menu.red .menu.blue -side left
pack .menu.help -side right


### Create the playing arena.

canvas .arena -width [get_const XMAX] -height [get_const YMAX] \
	-background black
pack .arena -padx 3 -pady 3

# Put the bases on their places.

set red_base \
	[.arena create oval \
	[expr [get_const PLAYER0_BASE_X] - [get_const PLAYER0_BASE_RADIUS]] \
	[expr [get_const PLAYER0_BASE_Y] - [get_const PLAYER0_BASE_RADIUS]] \
	[expr [get_const PLAYER0_BASE_X] + [get_const PLAYER0_BASE_RADIUS]] \
	[expr [get_const PLAYER0_BASE_Y] + [get_const PLAYER0_BASE_RADIUS]] \
	-fill $player0_color]
set blue_base \
	[.arena create oval \
	[expr [get_const PLAYER1_BASE_X] - [get_const PLAYER1_BASE_RADIUS]] \
	[expr [get_const PLAYER1_BASE_Y] - [get_const PLAYER1_BASE_RADIUS]] \
	[expr [get_const PLAYER1_BASE_X] + [get_const PLAYER1_BASE_RADIUS]] \
	[expr [get_const PLAYER1_BASE_Y] + [get_const PLAYER1_BASE_RADIUS]] \
	-fill $player1_color]


### Put status info, fancy buttons and labels to the bottom of the window.

frame .status

label .status.score -text " score = 0 - 0 " \
	-relief raised -borderwidth 1

frame .status.actions -relief sunken -border 2

frame .status.actions.steps
button .status.actions.steps.beginning \
	-bitmap @$minpantu_dir/tape_rewind.xbm \
	-command step_rewind -state disabled
button .status.actions.steps.previous \
	-bitmap @$minpantu_dir/tape_rwd_step.xbm \
	-command step_previous -state disabled
button .status.actions.steps.next \
	-bitmap @$minpantu_dir/tape_fwd_step.xbm \
	-command step_next
button .status.actions.steps.end \
	-bitmap @$minpantu_dir/tape_forward.xbm \
	-command step_end -state disabled
pack .status.actions.steps.beginning .status.actions.steps.previous \
	.status.actions.steps.next .status.actions.steps.end -side left

frame .status.actions.view
button .status.actions.view.play -bitmap @$minpantu_dir/tape_play.xbm \
	-command view_play
button .status.actions.view.stop -bitmap @$minpantu_dir/tape_stop.xbm \
	-command view_stop -state disabled
pack .status.actions.view.play .status.actions.view.stop -side left

pack .status.actions.steps .status.actions.view -side left -fill both -padx 5

label .status.minpantu -bitmap @$minpantu_dir/minpantu-small.xbm \
	-relief raised -borderwidth 1

pack .status.score -side left -fill both
pack .status.actions -side left -fill both -padx 10
pack .status.minpantu -side right -fill both
pack .status -side bottom -fill both -padx 3 -pady 3


### Make balls, but put them first in an invisible place.

# The weird syntax below is the way arrays are created in TCL.
set ball(0) 0

for {set i 0} {$i < [get_const NUMBER_OF_BALLS]} {incr i} {
    if {$i == 0} {
	set fc $player0_color
	set bc white
    } else { 
	if {$i == 1} {
	    set fc $player1_color
	    set bc white
	} else {
	    if {[expr $i % 2] == 0} {
		set fc black
		set bc $player0_color
	    } else {
		set fc black
		set bc $player1_color
	    }
	}
    }

    set ball($i) [.arena create oval 0 0 0 0 -fill $fc -outline $bc -width 2]
}

proc update_balls step {
    global ball

    for {set i 0} {$i < [get_const NUMBER_OF_BALLS]} {incr i} {
	.arena coords $ball($i) \
		[expr [ball x $i $step]-5] [expr [ball y $i $step]-5] \
		[expr [ball x $i $step]+5] [expr [ball y $i $step]+5]
    }
}


### Same for small white dots that represent intermediate states.

set dot(0) 0

for {set i 0} {$i < [get_const NUMBER_OF_BALLS]} {incr i} {
    for {set s 0} {$s < [expr [get_const STEPS_PER_PLY]-1]} {incr s} {
	set dot([expr $s * [get_const STEPS_PER_PLY] + $i]) \
		[.arena create rect 0 0 0 0 \
		-fill white -outline white -width 0]
    }
}


proc update_dots {} {
    global dot
    for {set i 0} {$i < [get_const NUMBER_OF_BALLS]} {incr i} {
	for {set s 0} {$s < [expr [get_const STEPS_PER_PLY]-1]} {incr s} {
	    .arena coords $dot([expr $s * [get_const STEPS_PER_PLY] + $i]) \
		    [expr [ball x $i $s]-1] [expr [ball y $i $s]-1] \
		    [expr [ball x $i $s]+1] [expr [ball y $i $s]+1]
	}
    }
}


proc hide_dots {} {
    global dot
    for {set i 0} {$i < [get_const NUMBER_OF_BALLS]} {incr i} {
	for {set s 0} {$s < [expr [get_const STEPS_PER_PLY]-1]} {incr s} {
	    .arena coords $dot([expr $s * [get_const STEPS_PER_PLY] + $i]) \
		    0 0 0 0
	}
    }
}


### User input handling.

# The interface can be in two possible states -- a smoothly moving
# view-only mode, or the stepwise mode where intermediate states are
# shown with small white dots and the user may be prompted for a move.

set is_viewing 0
set hit_enabled 0

set hit [.arena create rect 0 0 0 0 -outline $hit_color -width 2]
.arena lower $hit
# Move the bases even lower.
.arena lower $red_base
.arena lower $blue_base


proc update_inputs {} {
    global is_viewing
    global red_player blue_player human_player
    global hit hit_enabled hit_distance

    if {$is_viewing} {
	.status.actions.steps.beginning config -state disabled
	.status.actions.steps.previous config -state disabled
	.status.actions.steps.next config -state disabled
	.status.actions.steps.end config -state disabled
	.status.actions.view.play config -state disabled
	.status.actions.view.stop config -state normal
	set hit_enabled 0
	.arena coords $hit 0 0 0 0
    } else {

	.status.actions.steps.next config -state normal
	.status.actions.view.play config -state normal
	if {(([history_turn] == "RED") \
		&& ($red_player == $human_player)) \
		|| (([history_turn] == "BLUE") \
		&& ($blue_player == $human_player))} {
	    set hit_enabled 1
	    if {[history_turn] == "RED"} {
		set ball 0
	    } else {
		set ball 1
	    }
	    .arena coords $hit \
		    [expr [ball x $ball -current]-$hit_distance] \
		    [expr [ball y $ball -current]-$hit_distance] \
		    [expr [ball x $ball -current]+$hit_distance] \
		    [expr [ball y $ball -current]+$hit_distance]
	    if {[history_is_at_end]} {
		.status.actions.steps.next config -state disabled
		.status.actions.view.play config -state disabled
	    }
	} else {
	    set hit_enabled 0
	    .arena coords $hit 0 0 0 0
	}

	if {[history_is_at_beginning]} {
	    .status.actions.steps.beginning config -state disabled
	    .status.actions.steps.previous config -state disabled
	} else {
	    .status.actions.steps.beginning config -state normal
	    .status.actions.steps.previous config -state normal
	}
	if {[history_is_at_end]} {
	    .status.actions.steps.end config -state disabled
	} else {
	    .status.actions.steps.end config -state normal
	}
	.status.actions.view.stop config -state disabled
    }

    # In addition to updating inputs, update the score table.
    .status.score config -text [ball_score]
}


# Bind a button press on the underlying `.arena'-canvas to handle.  It is
# be possible to bind the handler to the hit-item itself, but this would
# moves that correspond to button presses on places obscured by other items
# on the canvas. 

bind .arena <ButtonPress-1> "user_move %x %y"

proc user_move {x y} {
    global hit_enabled hit_distance

    if {$hit_enabled} {
	if {[history_turn] == "RED"} {
	    set ball 0
	} else {
	    set ball 1
	}
	set rx [expr $x-[ball x $ball -current]]
	set ry [expr $y-[ball y $ball -current]]
	if {($rx <= $hit_distance) && ($rx >= -$hit_distance) \
		&& ($ry <= $hit_distance) && ($ry >= -$hit_distance)} {
	    set_user_move $rx $ry $hit_distance
	    history_next
	    update_balls -current
	    update_dots
	    update_inputs
	}
    }
}


proc new_game {} {
    history_new
    history_next
    update_balls -current
    update_dots
    update_inputs
}

new_game

proc step_rewind {} {
    history_rewind
    update_balls -current
    update_dots
    update_inputs
}

proc step_previous {} {
    history_previous
    update_balls -current
    update_dots
    update_inputs
}

proc step_next {} {
    global red_player blue_player
    if {[history_is_at_end]} {
	computer_move $red_player $blue_player
    }
    history_next
    update_balls -current
    update_dots
    update_inputs
}

proc step_end {} {
    history_end
    update_balls -current
    update_dots
    update_inputs
}


# Smooth "real-time" replaying of the game.

set view_step_counter 0

# This is called every 25th millisecond when the smooth motion is enabled.

proc do_view_play {} {
    global is_viewing view_step_counter
    global red_player blue_player human_player

    if {$view_step_counter == [expr [get_const STEPS_PER_PLY]-1]} {
	update_balls -current
	if {$is_viewing} {
	    set view_step_counter 0
	    if {[history_is_at_end]} {
		if {((([history_turn] == "RED") \
			&& ($red_player == $human_player)) \
			|| (([history_turn] == "BLUE") \
			&& ($blue_player == $human_player)))} {
		    set is_viewing 0
		    update_dots
		    update_inputs
		} else {
		    .status.score config -text [ball_score]
		    computer_move $red_player $blue_player
		    history_next
		    after 25 do_view_play
		}
	    } else {
		.status.score config -text [ball_score]
		history_next
		after 25 do_view_play
	    }
	} else {
	    update_dots
	    update_inputs
	}
    } else {
	update_balls $view_step_counter
	incr view_step_counter
	after 25 do_view_play
    }
}

# This initiates the repeated calling of `do_view_play'.

proc view_play {} {
    global is_viewing

    set is_viewing 1
    hide_dots
    update_inputs
    after 25 do_view_play
}

# This discontinues the repeated calling of `do_view_play'.

proc view_stop {} {
    global is_viewing

    set is_viewing 0
}


### Help texts.

proc name_help {{w .nh}} {
    global minpantu_dir

    catch {destroy $w}
    toplevel $w
    wm title $w "What's \"minpantu\"?"
    label $w.minpantu -bitmap @$minpantu_dir/minpantu-big.xbm
    pack $w.minpantu -pady 5
    button $w.ok -text "Dismiss" -command "destroy $w"
    pack $w.ok -side bottom -pady 3
    text $w.t -relief sunken -yscrollcommand "$w.s set" -setgrid true
    scrollbar $w.s -command "$w.t yview"
    pack $w.s -side right -fill y
    pack $w.t -expand yes -fill both

    $w.t insert 0.0 \
{Minpantu is the name of a game I was asked to program as an example
of a non-trivial programming project for a basic course of
(C-)programming in the Helsinki University of Technology.  See
http://www.hut.fi/tik-76.102/ In time the game proved to be an
interesting one, and I also spent some time to develop a computer
opponent and a gaudy interface for WWW and Tk.

The word "minpantu" is Tamil and means 'electric ball'.  I chose the
name, and the writing, because it describes well the funny
trajectories of the balls.

The source code is released under GNU Copyleft so that it can be
viewed and studied by anyone.  If you have any questions, suggestions,
improvements, or you want to report a bug, send mail to the author:

    Kenneth Oksanen
    http://www.cs.hut.fi/~cessu
    cessu@cs.hut.fi}
}

proc rule_help {{w .nh}} {
    global minpantu_dir

    catch {destroy $w}
    toplevel $w
    wm title $w "Rules of Minpantu"
    label $w.minpantu -bitmap @$minpantu_dir/minpantu-big.xbm
    pack $w.minpantu -pady 5
    button $w.ok -text "Dismiss" -command "destroy $w"
    pack $w.ok -side bottom -pady 3
    text $w.t -relief sunken -yscrollcommand "$w.s set" -setgrid true
    scrollbar $w.s -command "$w.t yview"
    pack $w.s -side right -fill y
    pack $w.t -expand yes -fill both

    $w.t insert 0.0 \
{The playing field contains 
  - two hit balls; the small red and blue balls with white borders
  - four goal balls; the small black balls with red or blue borders
  - two bases; the large red and blue disks.
The players assign a speed and direction to their hit balls at regular
intervals.  A player is rewarded one goal every time a goal ball
enters the opponent's base.  The objective of the game is to score
more goals than the opponent.

The objective of the game is made considerably difficult by the
various forces that affect the hit and goal balls.
  - All balls repulse each other.  The repulsive force of hit balls is 
    stronger than the repulsive force of the goals balls.  The
    repulsive force is proportional to the inverse of the square of
    the distance between the balls.
  - The walls of the playing field repulse the balls.  The repulsive
    force is inversely proportional to the distance from the wall.
  - The bases induce a rotating force field that affects the goal
    balls, but not the hit balls.  The red base tries to rotate goal
    balls with a red border clockwise, and goal balls with a blue
    border counter-clockwise.  The blue base induces a force field
    that rotates to the opposite direction.  The strength of the force
    field is inversely proportional to the distance of the wall.
  - Rapidly moving objects experience friction.
  - Finally, players' own bases repulse their hit balls.  This makes
    it more difficult for the players to play exclusively defensive
    play by nesting in their own bases and pushing away approaching
    goal balls.
I suggest you first familiarize yourself to the forces by assigning
both players to 'Nobody' from the menus 'Red Player' and 'Blue
Player', click the }
    button $w.t.play -bitmap @$minpantu_dir/tape_play.xbm
    $w.t window create end -window $w.t.play
    $w.t insert end \
{-button, and watch how the balls behave.

The prime directive for a strong performance in Minpantu is: Use the
Forces, Luke!}
}

proc play_help {{w .nh}} {
    global minpantu_dir

    catch {destroy $w}
    toplevel $w
    wm title $w "How to Play"
    label $w.minpantu -bitmap @$minpantu_dir/minpantu-big.xbm
    pack $w.minpantu -pady 5
    button $w.ok -text "Dismiss" -command "destroy $w"
    pack $w.ok -side bottom -pady 3
    text $w.t -relief sunken -yscrollcommand "$w.s set" -setgrid true
    scrollbar $w.s -command "$w.t yview"
    pack $w.s -side right -fill y
    pack $w.t -expand yes -fill both

    $w.t insert 0.0 \
{From the menus 'Red Player' and 'Blue Player' you can choose the
identity of the competitors.  'Computer Level X' means that the
computer, using a set of interesting methods, tries to find a good
move for the competitor it represents, and the higher 'X' is, the
better the moves will hopefully be.  'Random' means a player that
makes random moves, and 'Nobody' means that hit ball never makes a
change in its present speed.

If you choose 'Human' to play for either competitors, or both
competitors, you will be prompted for your move by a green rectangle
around your hit ball.  Clicking inside the rectangle will assign the
hit ball a speed that is relative to the click's location within the
rectangle.

The program memorizes all the moves made by the players.  By clicking
the }
    button $w.t.rewind -bitmap @$minpantu_dir/tape_rewind.xbm
    $w.t window create end -window $w.t.rewind
    $w.t insert end \
{-button below the arena you can move back to the start of the
game, and correspondlingly by clicking the }
    button $w.t.forward -bitmap @$minpantu_dir/tape_forward.xbm
    $w.t window create end -window $w.t.forward
    $w.t insert end \
{-button you can move 
to the end of the game played so far.  By clicking the buttons 
}
    button $w.t.rwd_step -bitmap @$minpantu_dir/tape_rwd_step.xbm
    $w.t window create end -window $w.t.rwd_step
    $w.t insert end { and }
    button $w.t.fwd_step -bitmap @$minpantu_dir/tape_fwd_step.xbm
    $w.t window create end -window $w.t.fwd_step
    $w.t insert end \
{ you can move one half-move (a.k.a. 'ply') backwards and 
forwards, i.e. to the point of time where either competitor previously
hit his ball.  At any point you can change the player, and if the
player in turn is 'Human', you can hit by clicking within the green
rectangle.  Note, however, that a change in the middle of a game will
unrevocably erase the rest of the game from the game history.

You can also have a smooth real-time animation of the played game by
clicking the }
    button $w.t.play -bitmap @$minpantu_dir/tape_play.xbm
    $w.t window create end -window $w.t.play
    $w.t insert end \
{-button, and stop the animation with }
    button $w.t.stop -bitmap @$minpantu_dir/tape_stop.xbm
    $w.t window create end -window $w.t.stop
    $w.t insert end \
{.  The animation
can extend the played game, but it will stop at least when a 'Human'
competitor is in turn in a position that hasn't yet been played.

Finally, the items 'Save' and 'Load' in the 'File'-menu can be used to
save and load the game.  Unfortunately the file name is fixed to
'game.mpt', but this restriction may be lifted as soon as a file
browser is incorporated into the program.  'New' can be used to start
a new game with a new initial position.}
}
