/* This file is part of alpha (= Alpha Loves to Perform Hasty Assembler).
 * See file COPYRIGHT for pertinent copyright notices.
 */


#ifndef INCL_EXE_H
#define INCL_EXE_H 1


#ifndef __GNUC__
#error GNU cc required to compile alpha.
#endif


/* The versions to be compiled are:
 *   o FAST = perform execution of everything as fast as possible.
 *   o DEBUG = performs checks e.g. on memory references, divisions by zero 
 *     etc.  Provide a command line for e.g. tracing and undo facility.
 *   o AGGRESSIVE = DEBUG + but with some additional features exposing more
 *     possible compiler erros, and no undo.
 */


#include <string.h>


typedef union word_t {
  /* This is the label a.k.a. computed goto of the instruction sequence
   * which performs the desired function.  This is much faster than using
   * function pointers, but requires gcc.
   */
  void *block;
  /* This pointer is used to refer to a comment string for the insn or a
   * label string.  Labels will be replaced with 'ptr's. 
   */
  char *string;
  /* Constant values, e.g. immediates. XXX Note that we are using
   * host's word length, not necessarily 64-bit words.  This is not
   * according UICC specs, but will do for a while.
   */
  long integer;
  /* This is what labels will be replaced with, direct pointers to next insn,
   * but this may also correspond to a memory location (static or register).
   */
  union word_t *ptr;
} word_t;


/* Define insn block indexes in the computed goto table.
 */

typedef enum {
#define INSN(mnemonic, compileblock, checkblock, executeblock, makeundoblock, undoblock) \
  mnemonic,
#include "insns.h"
#undef INSN
  MNEMONICS,
} mnemonic_t;


/* This array, initialized in exe.c, maps insn mnemonics to computed gotos
 * of the compiling block.
 */
extern void *compile_block[];


extern word_t *code;
extern int current_code_index;
/* Increase the code vector by one code_t.
 */
extern void grow_code(void);

#define ADD_WORD(field, value) \
  do { \
    grow_code(); \
    code[current_code_index].field = (value); \
    current_code_index++; \
  } while (0)

#ifdef FAST
#define ADD_COMMENT(s) /* */
#else
/* XXX Improve these macros! */
#define ADD_COMMENT(mnemonic) \
  do { \
    char buf[100]; \
    sprintf(buf, "%d:\t%s", asm_lineno, mnemonic); \
    ADD_WORD(string, strdup(buf)); \
  } while (0)
#endif


#define ADD_CODE_LABEL(lbl_string) \
  define_code_label(strdup(lbl_string), current_code_index)
#define ADD_LABEL_REF(lbl_string) ADD_WORD(string, strdup(lbl_string))
#define ADD_MNEMONIC(mnemonic) ADD_WORD(block, compile_block[(int) mnemonic])
#define ADD_SRC_REG(reg_num) ADD_WORD(ptr, &reg[reg_num])
#define ADD_DST_REG(reg_num) ADD_WORD(ptr, &reg[reg_num == 31 ? 32 : reg_num])
#define ADD_IMM(imm) ADD_WORD(integer, imm)


/* Memory configuration.  Strings are allocated dynamically and can't be
 * manipulated by UICC programs directly. 
 */
#define MEM_WORDS 131072 /* 1 megabyte for static data and stack. */
#define MIN_STACK_MEM_WORDS 8192 /* words again. */
extern word_t mem[MEM_WORDS];

word_t *make_static_array(int words);


/* This array contains the registers of the Alpha processor, plus
 * one which is used instead of register zero ($31) when writing to.
 */
extern word_t reg[32+1];


/* The routines below assumes the given 'char *label's are safe memory. */
extern void define_code_label(char *label, int code_offset);
extern void define_mem_label(char *label, word_t *mem);


extern int asm_lineno;


#ifndef FAST
typedef enum { 
  RUN_COMMAND,
  STEP_COMMAND,
  BACK_COMMAND,
} cli_command_t;
extern cli_command_t cli_command;
extern int cli_argument;
extern int tracing;
#endif


#endif
