/* This file is part of alpha (= Alpha Loves to Perform Hasty Assembler).
 * See file COPYRIGHT for pertinent copyright notices.
 */


#include <stdio.h>
#include "exe.h"


void *compile_block[MNEMONICS];


word_t *code = NULL;
int current_code_index = 0;
static int current_code_size = 0;

void grow_code(void)
{
  if (current_code_index + 1 >= current_code_size) {
    if (current_code_size == 0)
      current_code_size = 256;
    else
      current_code_size *= 2;
    code = (word_t *) realloc(code, current_code_size * sizeof(word_t));
    if (code == NULL) {
      fprintf(stderr, "Failed to allocate %d instructions.\n",
	      current_code_size);
      exit(1);
    }
  }
}


int check_code(word_t *p)
{
  if (p < code || p >= &code[current_code_size]) {
    fprintf(stderr, "Trying to jump to a nonexistent place.\n");
    return 0;
  }
  return 1;
}


/* XXX All data structures are currently lists.  This is not good
 * enough.  Rewrite them using hash tables.
 */

typedef struct label_t {
  struct label_t *next;
  char *label_name;
  enum { CODE_LABEL, MEM_LABEL } label_type;
  union {
    word_t *ptr;
    /* Have to use code_offset instead of 'word_t *' since the code may move. 
     */
    int code_offset;
  } u;
} label_t;

label_t *labels = NULL;


void define_code_label(char *label, int code_offset)
{
  label_t *l = (label_t *) malloc(sizeof(label_t));
  if (l == NULL) {
    fprintf(stderr, "Failed to allocate code label descriptor.\n");
    exit(1);
  }
  l->next = labels;
  l->label_name = label;
  l->label_type = CODE_LABEL;
  l->u.code_offset = code_offset;
  labels = l;
}


void define_mem_label(char *label, word_t *ptr)
{
  label_t *l = (label_t *) malloc(sizeof(label_t));
  if (l == NULL) {
    fprintf(stderr, "Failed to allocate mem label descriptor.\n");
    exit(1);
  }
  l->next = labels;
  l->label_name = label;
  l->label_type = MEM_LABEL;
  l->u.ptr = ptr;
  labels = l;
}


word_t *refer_label(char *s)
{
  label_t *l;
  for (l  = labels; l != NULL; l = l->next)
    if (l->label_name == s /* Optimization */ 
	|| strcmp(l->label_name, s) == 0)
      if (l->label_type == CODE_LABEL)
	return &code[l->u.code_offset];
      else
	return l->u.ptr;
  fprintf(stderr, "Unknown label '%s' referred.\n", s);
  exit(1);
}


/* String memory.   XXX Replace lists with hash tables. */

#ifndef FAST
typedef struct string_t {
  char *s;
  struct string_t *next;
} string_t;

static string_t *valid_strings = NULL;


void declare_string(char *s)
{
  string_t *p = (string_t *) malloc(sizeof(string_t));
  if (p == NULL) {
    fprintf(stderr, "Out of memory in 'declare_string'\n");
    exit(1);
  }
  p->s = s;
  p->next = valid_strings;
  valid_strings = p;
}


void discard_string(char *s)
{
  string_t *p = valid_strings, **pp = &valid_strings;
  while (p != NULL) {
    if (p->s == s) {		/* SIC! */
      *pp = p->next;		/* Remove 'p' from the list. */
      free(p);
      return;
    }
    pp = &p->next;
    p = p->next;
  }
  fprintf(stderr,
	  "Probable simulator error: trying to discard invalid string.\n");
  exit(1);
}


int check_string(char *s)
{
  string_t *p = valid_strings;
  while (p != NULL) {
    if (p->s == s)		/* SIC! */
      return 1;
    p = p->next;
  }
  fprintf(stderr, "Trying to refer to a non-existant string at 0x%08x\n", s);
  return 0;
}
#endif /* not FAST */


/* Memory, or more precisely static arrays and stack.  Strings are allocated
 * by other means and used only through library routines.
 */

word_t mem[MEM_WORDS];
static int mem_watermark = MEM_WORDS - 1;

/* Stack grows downwards starting from the point where the arrays end. */

word_t *make_static_array(int words)
{
  mem_watermark -= words;
  if (mem_watermark < MIN_STACK_MEM_WORDS) {
    fprintf(stderr, 
	    "Out of memory, could not allocate %d words static data.\n",
	    words);
    exit(1);
  }
  return &mem[mem_watermark];
}

#ifndef FAST
static inline word_t *check_addr(word_t *p)
{
  if ((((unsigned int) p) & 0x3) != 0) {
    fprintf(stderr, 
	    "Memory alignment error, trying to use address 0x%x\n",
	    p);
    exit(1);
  }
  if (p < mem  || p > &mem[MEM_WORDS-1]) {
    fprintf(stderr, "Memory error, trying to use address 0x%x\n", p);
    exit(1);
  }
  return p;
}
#endif


/* Registers. */

word_t reg[32+1];

#define reg_v0 reg[0]

#define reg_t0 reg[1]
#define reg_t1 reg[2]
#define reg_t2 reg[3]
#define reg_t3 reg[4]
#define reg_t4 reg[5]
#define reg_t5 reg[6]
#define reg_t6 reg[7]
#define reg_t7 reg[8]
#define reg_t8 reg[22]
#define reg_t9 reg[23]
#define reg_t10 reg[24]
#define reg_t11 reg[25]

#define reg_s0 reg[9]
#define reg_s1 reg[10]
#define reg_s2 reg[11]
#define reg_s3 reg[12]
#define reg_s4 reg[13]
#define reg_s5 reg[14]
#define reg_s6 reg[15]

#define reg_a0 reg[16]
#define reg_a1 reg[17]
#define reg_a2 reg[18]
#define reg_a3 reg[19]
#define reg_a4 reg[20]
#define reg_a5 reg[21]

#define reg_ra reg[26]
#define reg_at reg[28]
#define reg_gp reg[29]
#define reg_sp reg[30]


/* Undo log. */

#ifdef DEBUG

typedef struct undo_info_t {
  /* Computed goto of the piece of code that is to be executed when 
   * undoing.
   */
  void *undoer;

  /* Pointer to the word and the old value of the word that was
   * changed by the instruction. 
   */
  word_t *address;
  word_t old_value;
  
  /* Some "instructions" do more, e.g. free or copy data.  This
   * pointer can be used to store that information or to point to some
   * additional undo information. 
   */
  void *undo_extra;

  /* The program counter of the instruction executed before this
   * instruction.
   */
  word_t *pc;

  struct undo_info_t *old;
} undo_info_t;

static undo_info_t *undo_info;

#endif /* DEBUG */

#ifndef FAST
int tracing = 1;
#endif


/* The compilation and execution of the loaded program. */

extern char *prog_name;
extern int number_of_arguments;
extern char **arg_strings;

#ifndef FAST
cli_command_t cli_command;
int cli_argument;
extern int cliparse();
#endif

void execute(int only_init)
{
  word_t *pc = code;
#ifndef FAST
  int running;
#ifdef DEBUG
  undo_info_t *undo_tmp;
#endif
#endif

  if (only_init) {
    /* Initialize the 'compile_block' table. */
#define INSN(mnemonic, compileblock, checkblock, executeblock, makeundoblock, undoblock) \
    compile_block[mnemonic] = &&mnemonic ## _compile;
#include "insns.h"
#undef INSN
    
    /* Define library routines as functions of one dedicated instruction. */
#define DEFINE_UICC_BUILTIN(mnemonic) \
  ADD_CODE_LABEL("UICC_" #mnemonic); \
  ADD_MNEMONIC(UICC_ ## mnemonic); \
  ADD_COMMENT("UICC_" #mnemonic)
    DEFINE_UICC_BUILTIN(ReadInt);
    DEFINE_UICC_BUILTIN(ReadString);
    DEFINE_UICC_BUILTIN(WriteInt);
    DEFINE_UICC_BUILTIN(WriteString);
    DEFINE_UICC_BUILTIN(AllocateString);
    DEFINE_UICC_BUILTIN(FreeString);
    DEFINE_UICC_BUILTIN(StringToInt);
    DEFINE_UICC_BUILTIN(IntToString);
    DEFINE_UICC_BUILTIN(StringLength);
    DEFINE_UICC_BUILTIN(StringCompare);
    DEFINE_UICC_BUILTIN(CopyIntoString);
    DEFINE_UICC_BUILTIN(CopyFromString);
    DEFINE_UICC_BUILTIN(Arguments);
    DEFINE_UICC_BUILTIN(Argument);
    DEFINE_UICC_BUILTIN(ProgramName);
    DEFINE_UICC_BUILTIN(exit);
#undef DEFINE_UICC_BUILTIN

    return;
  }

  /* This loop "compiles" the created insns. 
   */
  while (pc < &code[current_code_index]) {
    goto *pc[0].block;
  return_from_compile_block:
  }

  /* Set stack pointer, program counter, and return address register
   * ready for execution.
   */
  reg_sp.ptr = &mem[mem_watermark];
  reg_ra.ptr = refer_label("UICC_exit");
  pc = refer_label("UICC_Main");

#ifdef FAST
  goto *pc->block;

  /* This is the label where the insn 'UICC_exit' jumps to. */
 do_exit:
  return;
#else
  /* Uh... Sorry for the gotos. */
  running = 1;
 cli_next:
  printf("> ");
  cliparse();

  if (cli_command == RUN_COMMAND || cli_command == STEP_COMMAND) {
    if (!running) {
      printf("Restarting program.\n");
      reg_sp.ptr = &mem[mem_watermark];
      reg_ra.ptr = refer_label("UICC_exit");
      pc = refer_label("UICC_Main");
#ifdef DEBUG
      undo_info = NULL;
#endif
      /* XXX Should free the log. This is WRONG! */
    }
    goto *pc->block;
    /*NOTREACHED*/
  return_from_execute_block:
    if (cli_command == RUN_COMMAND)
      goto *pc->block;
    goto cli_next;
    
  do_exit:
    printf("Program finished.\n");
    running = 0;
    goto cli_next;
  } 
#ifdef DEBUG
  if (cli_command == BACK_COMMAND) {
    if (undo_info != NULL)
      goto *undo_info->undoer;
    else
      printf("No undo log.\n");

  return_from_undo_block:
    goto cli_next;
  }
#endif
#endif


#ifdef FAST
#define INSN(mnemonic, compileblock, checkblock, executeblock, makeundoblock, undoblock) \
  mnemonic ## _compile: \
    pc[0].block = &&mnemonic ## _execute; \
    compileblock; \
    goto return_from_compile_block; \
  mnemonic ## _execute: \
    /* printf("pc = 0x%08x, insn = %s\n", pc, #mnemonic); */ \
    executeblock; \
    goto *pc[0].block;
#endif

#ifdef AGGRESSIVE
#define INSN(mnemonic, compileblock, checkblock, executeblock, makeundoblock, undoblock) \
  mnemonic ## _compile: \
    pc[0].block = &&mnemonic ## _execute; \
    pc++; \
    compileblock; \
    goto return_from_compile_block; \
  mnemonic ## _execute: \
    /* printf("pc = 0x%08x, insn = %s\n", pc, #mnemonic); */ \
    pc++; /* Skip over comment */ \
    if (tracing) \
      printf("%s\n", pc[0].string); \
    checkblock; \
    executeblock; \
    goto return_from_execute_block;
#endif

#ifdef DEBUG
#define INSN(mnemonic, compileblock, checkblock, executeblock, makeundoblock, undoblock) \
  mnemonic ## _compile: \
    pc[0].block = &&mnemonic ## _execute; \
    pc++; \
    compileblock; \
    goto return_from_compile_block; \
  mnemonic ## _execute: \
    /* printf("pc = 0x%08x, insn = %s\n", pc, #mnemonic); */ \
    pc++; /* Skip comment. */ \
    if (tracing) \
      printf("%s\n", pc[0].string); \
    checkblock; \
    undo_tmp = (undo_info_t *) malloc(sizeof(undo_info_t)); \
    undo_tmp->pc = pc - 1; \
    undo_tmp->undoer = && mnemonic ## _undo; \
    undo_tmp->old = undo_info; \
    undo_info = undo_tmp; \
    makeundoblock; \
    executeblock; \
    goto return_from_execute_block; \
  mnemonic ## _undo: \
    undoblock; \
    pc = undo_tmp->pc; \
    undo_tmp = undo_info->old; \
    free(undo_info); \
    undo_info = undo_tmp; \
    goto return_from_undo_block;
#endif /* DEBUG */

#include "insns.h"
#undef INSN

}
